<?php
// api/feed.php
declare(strict_types=1);

ini_set('display_errors', '1');
error_reporting(E_ALL);
header('Content-Type: application/json; charset=utf-8');

require_once dirname(__DIR__) . '/vendor/autoload.php';
require_once __DIR__ . '/config.php';

use Google\Cloud\Translate\V2\TranslateClient;
use Google\Cloud\TextToSpeech\V1\TextToSpeechClient;
use Google\Cloud\TextToSpeech\V1\SynthesisInput;
use Google\Cloud\TextToSpeech\V1\VoiceSelectionParams;
use Google\Cloud\TextToSpeech\V1\AudioConfig;
use Google\Cloud\TextToSpeech\V1\AudioEncoding;

function out(int $code, array $payload): never {
  http_response_code($code);
  echo json_encode($payload, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
  exit;
}
function sanitize_channel(string $raw): array {
  $raw = trim($raw);
  if ($raw === '' || !is_numeric($raw)) return [null, null, 'channel invalide'];
  $f = (float)$raw;
  if ($f < 0 || $f > 111.11) return [null, null, 'channel hors limites'];
  $norm = number_format($f, 2, '.', '');
  $file = str_replace('.', '_', $norm);
  return [$norm, $file, null];
}
/** simple POST JSON */
function http_post_json(string $url, array $headers, array $jsonBody): array {
  $ch = curl_init($url);
  $hdrs = []; foreach ($headers as $k=>$v) $hdrs[] = $k.': '.$v;
  curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_POST => true,
    CURLOPT_HTTPHEADER => $hdrs,
    CURLOPT_POSTFIELDS => json_encode($jsonBody, JSON_UNESCAPED_UNICODE),
    CURLOPT_HEADER => false,
  ]);
  $body = curl_exec($ch);
  $err  = curl_error($ch);
  $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
  curl_close($ch);
  return ['ok'=>($err==='' && $code>=200 && $code<300), 'code'=>$code, 'err'=>$err, 'body'=>$body];
}
/** TTS cache path */
function tts_cached_path(string $sessionId, string $eventId, string $tgt, string $provider): string {
  @mkdir(TTS_PUBLIC_DIR . '/' . $sessionId, 0775, true);
  $safeProv = preg_replace('/[^a-z0-9_-]/i', '', $provider);
  $safeTgt  = preg_replace('/[^a-z0-9_-]/i', '', $tgt);
  return TTS_PUBLIC_DIR . '/' . $sessionId . '/' . $eventId . '-' . $safeTgt . '-' . $safeProv . '.mp3';
}
function tts_public_url(string $sessionId, string $eventId, string $tgt, string $provider): string {
  $safeProv = preg_replace('/[^a-z0-9_-]/i', '', $provider);
  $safeTgt  = preg_replace('/[^a-z0-9_-]/i', '', $tgt);
  return 'tts/' . $sessionId . '/' . $eventId . '-' . $safeTgt . '-' . $safeProv . '.mp3';
}
function translate_text(string $text, string $target): string {
  $gcpCred = getenv('GOOGLE_APPLICATION_CREDENTIALS') ?: (defined('GCP_CREDENTIALS') ? GCP_CREDENTIALS : null);
  $tr = new TranslateClient(['keyFilePath' => $gcpCred]);
  $res = $tr->translate($text, ['target' => $target, 'format'=>'text']);
  return $res['text'] ?? $text;
}
function synth_google(string $text, string $target, string $outPath): void {
  $gcpCred = getenv('GOOGLE_APPLICATION_CREDENTIALS') ?: (defined('GCP_CREDENTIALS') ? GCP_CREDENTIALS : null);
  $voiceLangMap = ['en'=>'en-US','fr'=>'fr-FR','ar'=>'ar-XA','es'=>'es-ES','de'=>'de-DE','it'=>'it-IT','pt'=>'pt-PT','ru'=>'ru-RU','tr'=>'tr-TR'];
  $voiceLang    = $voiceLangMap[strtolower($target)] ?? $target;

  $ttsClient = new TextToSpeechClient(['transport'=>'rest','credentials'=>$gcpCred]);
  $inputText = (new SynthesisInput())->setText($text);
  $voiceSel  = (new VoiceSelectionParams())->setLanguageCode($voiceLang);
  $audioCfg  = (new AudioConfig())->setAudioEncoding(AudioEncoding::MP3);
  $ttsResp   = $ttsClient->synthesizeSpeech($inputText, $voiceSel, $audioCfg);
  $audio     = $ttsResp->getAudioContent();
  if (method_exists($ttsClient, 'close')) $ttsClient->close();

  file_put_contents($outPath, $audio);
}
function synth_eleven(string $text, string $voiceId, string $outPath): bool {
  $api = defined('ELEVEN_API_KEY') ? ELEVEN_API_KEY : '';
  if (!$api) return false;
  $url = 'https://api.elevenlabs.io/v1/text-to-speech/'.rawurlencode($voiceId ?: '21m00Tcm4TlvDq8ikWAM');
  $res = http_post_json($url, [
    'accept' => 'audio/mpeg',
    'xi-api-key' => $api,
    'Content-Type'=> 'application/json'
  ], [
    'text'=>$text, 'model_id'=>'eleven_multilingual_v2',
    'voice_settings'=>['stability'=>0.5,'similarity_boost'=>0.75],
    'output_format'=>'mp3_44100_128'
  ]);
  if (!$res['ok'] || !is_string($res['body']) || strlen($res['body'])===0) return false;
  file_put_contents($outPath, $res['body']);
  return true;
}
function synth_openai(string $text, string $voice, string $outPath): bool {
  $api = defined('OPENAI_API_KEY') ? OPENAI_API_KEY : '';
  if (!$api) return false;
  $v = strtolower(trim($voice ?: 'alloy'));
  $map = ['allow'=>'alloy','aloy'=>'alloy','versa'=>'verse','vèrse'=>'verse']; if (isset($map[$v])) $v = $map[$v];
  $res = http_post_json('https://api.openai.com/v1/audio/speech', [
    'Authorization'=>'Bearer '.$api,'Accept'=>'audio/mpeg','Content-Type'=>'application/json'
  ], [
    'model'=>'gpt-4o-mini-tts','input'=>$text,'voice'=>$v,'format'=>'mp3'
  ]);
  if (!$res['ok'] || !is_string($res['body']) || strlen($res['body'])===0) return false;
  file_put_contents($outPath, $res['body']);
  return true;
}

/* --------------------------- Entrée --------------------------- */
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

$channelRaw    = $_GET['channel'] ?? null;
$since         = isset($_GET['since']) ? (float)$_GET['since'] : null;
$cursorNow     = isset($_GET['cursor']) && $_GET['cursor']==='now';
$limit         = max(0, (int)($_GET['limit'] ?? 50));
$excludeSpk    = strtolower(trim($_GET['exclude_speaker_id'] ?? ''));
$excludeSess   = strtolower(trim($_GET['exclude_session_id'] ?? ''));
$viewerIsHost  = (int)($_GET['viewer_is_host'] ?? 0) === 1;
$targetLang    = strtolower(trim($_GET['target_lang'] ?? 'en')); // ← langue choisie par L’AUDITEUR

if (!$channelRaw) out(400, ['ok'=>false,'error'=>'channel requis']);
[$channelNorm, $channelFile, $chanErr] = sanitize_channel($channelRaw);
if ($chanErr !== null) out(400, ['ok'=>false,'error'=>'channel invalide (0..111.11)']);

$feedPath = FEED_DIR . '/' . $channelFile . '.ndjson';
if (!is_file($feedPath)) out(200, ['ok'=>true, 'items'=>[], 'next_since'=>($since ?: microtime(true))]);

$lines = @file($feedPath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
if ($lines === false) out(500, ['ok'=>false, 'error'=>'Lecture du feed échouée']);

$now = microtime(true);
$startTs = $cursorNow ? $now : ($since ?: 0.0);

$items = [];
$maxTs = $startTs;
foreach ($lines as $line) {
  $ev = json_decode($line, true);
  if (!$ev || !isset($ev['ts'])) continue;

  if ($ev['ts'] <= $startTs) continue;
  if ($viewerIsHost && ($ev['speaker_role'] ?? '') === 'listener') continue; // l’hôte n’entend pas les auditeurs
  if ($excludeSpk && strtolower($ev['speaker_id'] ?? '') === $excludeSpk) continue;
  if ($excludeSess && strtolower($ev['session_id'] ?? '') === $excludeSess) continue;

  $maxTs = max($maxTs, (float)$ev['ts']);

  // 1) Traduction vers la langue DEMANDÉE par l’auditeur
  $srcText = trim((string)($ev['stt'] ?? ''));
  if ($srcText === '') continue;
  $translated = translate_text($srcText, $targetLang);

  // 2) TTS selon le provider/voix CHOISIS PAR L’HÔTE au moment du post
  $provider = strtolower($ev['tts_provider'] ?? (defined('TTS_DEFAULT_PROVIDER') ? TTS_DEFAULT_PROVIDER : 'google'));
  $voice    = $ev['tts_voice'] ?? null;

  $mp3Path  = tts_cached_path((string)$ev['session_id'], (string)$ev['id'], $targetLang, $provider);
  $mp3Url   = tts_public_url((string)$ev['session_id'], (string)$ev['id'], $targetLang, $provider);

  if (!is_file($mp3Path)) {
    // génère MP3 (avec fallback Google si besoin)
    $ok = false;
    if ($provider === 'elevenlabs') $ok = synth_eleven($translated, (string)$voice, $mp3Path);
    else if ($provider === 'openai') $ok = synth_openai($translated, (string)$voice, $mp3Path);

    if (!$ok) { // Google fallback
      synth_google($translated, $targetLang, $mp3Path);
      $provider = 'google';
      $voice    = $targetLang;
      // on n’écrit pas dans le feed, c’est propre à l’auditeur
    }
  }

  $items[] = [
    'ts'            => $ev['ts'],
    'id'            => $ev['id'],
    'session_id'    => $ev['session_id'],
    'channel'       => $ev['channel'],
    'speaker_id'    => $ev['speaker_id'],
    'speaker_role'  => $ev['speaker_role'],
    'source_lang'   => $ev['source_lang'],
    'target_lang'   => $targetLang,          // ← langue de L’AUDITEUR
    'text'          => $translated,          // ce que l’auditeur lit
    'tts_url'       => $mp3Url,              // ce que l’auditeur entend
    'tts_provider'  => $provider,            // réellement utilisé pour CE MP3
    'tts_voice'     => $voice
  ];

  if ($limit > 0 && count($items) >= $limit) break;
}

out(200, ['ok'=>true,'items'=>$items,'next_since'=>$maxTs > $startTs ? $maxTs : $now]);
