<?php
declare(strict_types=1);

// api/chunk.php
ini_set('display_errors', '1');
error_reporting(E_ALL);
header('Content-Type: application/json; charset=utf-8');

require_once dirname(__DIR__) . '/vendor/autoload.php';
require_once __DIR__ . '/config.php';

use Google\Cloud\Core\Exception\GoogleException;

// Speech-to-Text V2
use Google\Cloud\Speech\V2\Client\SpeechClient as SpeechClientV2;
use Google\Cloud\Speech\V2\RecognitionConfig as RecognitionConfigV2;
use Google\Cloud\Speech\V2\RecognitionFeatures as RecognitionFeaturesV2;
use Google\Cloud\Speech\V2\AutoDetectDecodingConfig as AutoDetectDecodingConfigV2;
use Google\Cloud\Speech\V2\ExplicitDecodingConfig as ExplicitDecodingConfigV2;
use Google\Cloud\Speech\V2\ExplicitDecodingConfig\AudioEncoding as AudioEncodingV2;
use Google\Cloud\Speech\V2\RecognizeRequest as RecognizeRequestV2;

function out(int $code, array $payload): never {
  http_response_code($code);
  echo json_encode($payload, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
  exit;
}
function get_project_id_from_credentials(?string $credPath): ?string {
  if (!$credPath || !is_file($credPath)) return null;
  $json = json_decode((string)@file_get_contents($credPath), true);
  return $json['project_id'] ?? null;
}
function sanitize_channel(string $raw): array {
  $raw = trim($raw);
  if ($raw === '' || !is_numeric($raw)) return [null, null, 'channel invalide'];
  $f = (float)$raw;
  if ($f < 0 || $f > 111.11) return [null, null, 'channel hors limites'];
  $norm = number_format($f, 2, '.', '');
  $file = str_replace('.', '_', $norm);
  return [$norm, $file, null];
}
function normalize_id(string $id): string {
  return strtolower(preg_replace('/[^a-z0-9-]/i', '', $id));
}
function detectEncoding(string $mime, string $filename): int {
  $m = strtolower($mime);
  $ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
  if (str_contains($m, 'ogg') || $ext === 'ogg' || $ext === 'oga') return AudioEncodingV2::OGG_OPUS;
  return AudioEncodingV2::WEBM_OPUS;
}

/* --------------------------- Préflight --------------------------- */
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

try {
  $seq           = isset($_POST['seq']) ? (int)$_POST['seq'] : null;
  $sessionId     = $_POST['session_id']   ?? null;
  $speakerIdRaw  = $_POST['speaker_id']   ?? null;
  $speakerRole   = $_POST['speaker_role'] ?? 'listener'; // "host" | "listener"
  $channelRaw    = $_POST['channel']      ?? null;
  $sourceLang    = $_POST['source_lang']  ?? 'fr-FR';

  // Choix TTS *déclaratif* (l’hôte décide du provider/voix à utiliser pour tous les auditeurs)
  $ttsProviderReq= strtolower($_POST['tts_provider'] ?? '');
  $ttsVoiceReq   = $_POST['tts_voice']    ?? null;

  if (!$sessionId || !preg_match('/^[a-f0-9-]{8,}$/i', $sessionId)) out(400, ['ok'=>false, 'seq'=>$seq, 'error'=>'session_id invalide']);
  if (!$speakerIdRaw) out(400, ['ok'=>false, 'seq'=>$seq, 'error'=>'speaker_id invalide']);
  $speakerId = normalize_id($speakerIdRaw);
  if ($speakerId === '') out(400, ['ok'=>false, 'seq'=>$seq, 'error'=>'speaker_id invalide']);
  $speakerRole = (strtolower($speakerRole) === 'host') ? 'host' : 'listener';

  if (!$channelRaw) out(400, ['ok'=>false, 'seq'=>$seq, 'error'=>'channel requis']);
  [$channelNorm, $channelFile, $chanErr] = sanitize_channel($channelRaw);
  if ($chanErr !== null) out(400, ['ok'=>false, 'seq'=>$seq, 'error'=>'channel invalide (0..111.11)']);

  if (!isset($_FILES['audio']['tmp_name'])) out(400, ['ok'=>false, 'seq'=>$seq, 'error'=>'Aucun fichier audio reçu']);

  /* ----------------------- Stockage chunk brut ---------------------- */
  $sessDir = SESSIONS_DIR . '/' . $sessionId;
  if (!is_dir($sessDir) && !@mkdir($sessDir, 0775, true)) out(500, ['ok'=>false, 'seq'=>$seq, 'error'=>'Impossible de créer le répertoire session']);

  $uploadName = $_FILES['audio']['name'] ?? 'chunk.webm';
  $mime       = $_FILES['audio']['type'] ?? 'application/octet-stream';
  $ext        = (str_contains($mime, 'ogg') || preg_match('/\.ogg$/i', $uploadName)) ? 'ogg' : 'webm';

  $seqName    = ($seq !== null) ? $seq : (int)(microtime(true)*1000);
  $tmp        = $_FILES['audio']['tmp_name'];
  $chunkPath  = $sessDir . '/' . $seqName . '.' . $ext;
  if (!@move_uploaded_file($tmp, $chunkPath)) {
    if (!@rename($tmp, $chunkPath)) out(500, ['ok'=>false, 'seq'=>$seq, 'error'=>'Impossible de stocker le chunk audio']);
  }

  $audioBytes = @file_get_contents($chunkPath);
  if ($audioBytes === false || $audioBytes === '') out(500, ['ok'=>false, 'seq'=>$seq, 'error'=>'Lecture du chunk échouée']);
  if (strlen($audioBytes) < 1024) out(200, ['ok'=>true,'seq'=>$seq,'channel'=>$channelNorm,'source_lang'=>$sourceLang,'text'=>'']);

  /* ----------------- Google STT v2 (reco) ----------------- */
  $gcpCred   = getenv('GOOGLE_APPLICATION_CREDENTIALS') ?: (defined('GCP_CREDENTIALS') ? GCP_CREDENTIALS : null);
  $projectId = get_project_id_from_credentials($gcpCred) ?: getenv('GOOGLE_CLOUD_PROJECT') ?: getenv('GCLOUD_PROJECT');
  if (!$projectId) out(500, ['ok'=>false, 'seq'=>$seq, 'error'=>'project_id introuvable (depuis la clé JSON)']);
  $recognizer = "projects/{$projectId}/locations/global/recognizers/_";

  if (!class_exists(SpeechClientV2::class)) out(500, ['ok'=>false, 'seq'=>$seq, 'error'=>'Client Speech V2 introuvable']);
  $speech = new SpeechClientV2();

  $features   = (new RecognitionFeaturesV2())->setEnableAutomaticPunctuation(true);
  $encGuessed = detectEncoding($mime, $uploadName);

  $makeConfig = function(?int $explicit) use ($features, $sourceLang): RecognitionConfigV2 {
    $cfg = new RecognitionConfigV2();
    if ($explicit !== null) {
      $dec = (new ExplicitDecodingConfigV2())
        ->setEncoding($explicit)->setSampleRateHertz(48000)->setAudioChannelCount(1);
      $cfg->setExplicitDecodingConfig($dec);
    } else {
      $cfg->setAutoDecodingConfig(new AutoDetectDecodingConfigV2());
    }
    return $cfg->setLanguageCodes([$sourceLang])->setModel('latest_long')->setFeatures($features);
  };

  $config  = $makeConfig($encGuessed);
  $request = (new RecognizeRequestV2())->setRecognizer($recognizer)->setConfig($config)->setContent($audioBytes);
  try { $response = $speech->recognize($request); }
  catch (GoogleException $eA) {
    $config = $makeConfig(null);
    $request = (new RecognizeRequestV2())->setRecognizer($recognizer)->setConfig($config)->setContent($audioBytes);
    $response = $speech->recognize($request);
  }

  $transcript = '';
  foreach ($response->getResults() as $res) {
    $alts = $res->getAlternatives();
    if (count($alts) > 0 && $alts[0]->getTranscript() !== '') {
      $transcript .= ($transcript === '' ? '' : ' ') . trim($alts[0]->getTranscript());
    }
  }

  // Normalise les choix de l’hôte (provider/voix)
  $provider = $ttsProviderReq ?: (defined('TTS_DEFAULT_PROVIDER') ? TTS_DEFAULT_PROVIDER : 'google');
  $voice    = $ttsVoiceReq ?: null;
  if ($provider === 'openai' && $voice) {
    $map = ['allow'=>'alloy','aloy'=>'alloy','versa'=>'verse','vèrse'=>'verse'];
    $vk = strtolower(trim($voice)); if (isset($map[$vk])) $voice = $map[$vk];
  }

  /* ------------------ Publication évènement ------------------ */
  if (!defined('FEED_DIR')) { define('FEED_DIR', __DIR__ . '/feeds'); @mkdir(FEED_DIR, 0775, true); }
  $feedPath = FEED_DIR . '/' . $channelFile . '.ndjson';

  $event = [
    'ts'           => microtime(true),
    'id'           => $sessionId . '-' . $seqName,
    'seq'          => $seqName,
    'session_id'   => strtolower($sessionId),
    'channel'      => $channelNorm,
    'speaker_id'   => $speakerId,
    'speaker_role' => $speakerRole,
    'source_lang'  => $sourceLang,
    'stt'          => $transcript,
    // choix TTS de l’hôte (appliqués lors du feed par auditeur)
    'tts_provider' => $provider,
    'tts_voice'    => $voice
  ];
  $line = json_encode($event, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES) . "\n";
  if ($fh = @fopen($feedPath, 'ab')) {
    if (function_exists('flock')) @flock($fh, LOCK_EX);
    @fwrite($fh, $line);
    if (function_exists('flock')) @flock($fh, LOCK_UN);
    @fclose($fh);
  }

  out(200, [
    'ok'=>true,'seq'=>$seq,'channel'=>$channelNorm,'source_lang'=>$sourceLang,
    'text'=>$transcript,'tts_provider'=>$provider,'tts_voice'=>$voice
  ]);

} catch (GoogleException $ge) {
  out(500, ['ok'=>false, 'seq'=>$seq ?? null, 'error'=>'GoogleException: '.$ge->getMessage()]);
} catch (Throwable $e) {
  out(500, ['ok'=>false, 'seq'=>$seq ?? null, 'error'=>'Exception: '.$e->getMessage()]);
}
