<?php
// api/config.php
declare(strict_types=1);

/**
 * Configuration centrale (chemins, CORS, limites d’upload, providers TTS).
 * - Google Cloud : utilise la variable d’environnement GOOGLE_APPLICATION_CREDENTIALS (déjà exportée).
 * - ElevenLabs & OpenAI : passe les clés via .htaccess (SetEnv) ou variables d’environnement.
 */

ini_set('display_errors', '1');
error_reporting(E_ALL);

// -------------------------------
// Chemins / Dossiers de travail
// -------------------------------
define('ROOT_DIR', dirname(__DIR__));            // /.../chanel.miva.app
define('API_DIR', __DIR__);                       // /.../chanel.miva.app/api
define('PUBLIC_DIR', ROOT_DIR . '/public');       // /.../chanel.miva.app/public

define('SESSIONS_DIR', API_DIR . '/sessions');    // Uploads audio (chunks)
define('TTS_PUBLIC_DIR', PUBLIC_DIR . '/tts');    // Sorties MP3 jouables côté web
define('FEED_DIR', API_DIR . '/feeds');           // Bus NDJSON par channel
define('CACHE_DIR', API_DIR . '/cache');          // Petits caches (ex: voix ElevenLabs)

@mkdir(SESSIONS_DIR, 0775, true);
@mkdir(TTS_PUBLIC_DIR, 0775, true);
@mkdir(FEED_DIR, 0775, true);
@mkdir(CACHE_DIR, 0775, true);

// --- Fallback ENV (utile si SetEnv n'est pas transmis à PHP-FPM) ---
$__gjson = getenv('GOOGLE_APPLICATION_CREDENTIALS');
if (!$__gjson || !is_readable($__gjson)) {
  // mets ici ton chemin ABSOLU vers le JSON :
  $__gjson = '/home/miva/FileDocs/miva-connect-473519-75b4e78ab97f.json';
  if (is_readable($__gjson)) {
    putenv('GOOGLE_APPLICATION_CREDENTIALS=' . $__gjson);
  }
}

if (!getenv('GOOGLE_CLOUD_PROJECT')) {
  putenv('GOOGLE_CLOUD_PROJECT=miva-connect-473519');
}

// -------------------------------
// CORS + limites HTTP utiles
// -------------------------------
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Headers: Content-Type');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');

@ini_set('upload_max_filesize', '50M');
@ini_set('post_max_size', '50M');
@ini_set('max_execution_time', '120');
@ini_set('memory_limit', '256M');

// -------------------------------
// Google Cloud (clé déjà exportée)
// -------------------------------
// Ne rien forcer ici si GOOGLE_APPLICATION_CREDENTIALS est déjà défini.
// Pour forcer un chemin local (déconseillé si ENV déjà en place) :
// putenv('GOOGLE_APPLICATION_CREDENTIALS=' . '/chemin/vers/ton-projet.json');
// define('GCP_CREDENTIALS', '/chemin/vers/ton-projet.json');

// -------------------------------
// Providers TTS (choix & clés)
// -------------------------------
// Provider par défaut si le front n’envoie rien : 'google' | 'elevenlabs' | 'openai'
define('TTS_DEFAULT_PROVIDER', getenv('TTS_DEFAULT_PROVIDER') ?: 'google');

// ElevenLabs : accepte ELEVENLABS_API_KEY ou ELEVEN_API_KEY
$__eleven = getenv('ELEVENLABS_API_KEY') ?: getenv('ELEVEN_API_KEY') ?: '';
define('ELEVEN_API_KEY', $__eleven);

// OpenAI (pour TTS gpt-4o-mini-tts)
define('OPENAI_API_KEY', getenv('OPENAI_API_KEY') ?: '');

// -------------------------------
// Helper debug facultatif
// -------------------------------
if (!function_exists('cfg_info')) {
  function cfg_info(): array {
    $mask = function (?string $k): ?string {
      if (!$k) return null;
      $len = strlen($k);
      if ($len <= 8) return str_repeat('•', $len);
      return substr($k, 0, 4) . '…' . substr($k, -4);
    };

    return [
      'autoload.path' => ROOT_DIR . '/vendor/autoload.php',
      'config.path'   => __FILE__,
      'php.version'   => PHP_VERSION,
      'php.sapi'      => PHP_SAPI,
      'env.GOOGLE_APPLICATION_CREDENTIALS' => getenv('GOOGLE_APPLICATION_CREDENTIALS') ?: 'NOT SET',
      'dirs' => [
        'SESSIONS_DIR'   => SESSIONS_DIR,
        'TTS_PUBLIC_DIR' => TTS_PUBLIC_DIR,
        'FEED_DIR'       => FEED_DIR,
        'CACHE_DIR'      => CACHE_DIR,
      ],
      'tts' => [
        'default'          => TTS_DEFAULT_PROVIDER,
        'eleven.keySet'    => ELEVEN_API_KEY !== '',
        'eleven.keyMasked' => $mask(ELEVEN_API_KEY),
        'openai.keySet'    => OPENAI_API_KEY !== '',
        'openai.keyMasked' => $mask(OPENAI_API_KEY),
      ],
    ];
  }
}
