// public/app.js
// =======================================================
// Niva Connect — PTT + STT + feed multi-langues
// - Hôte serveur (claim/release) + autoriser/interdire la parole
// - Mute ciblé d’un auditeur
// - Fix iOS/Safari (overlay d’activation audio cliquable)
// - Choix TTS (Google/OpenAI/ElevenLabs) contrôlé par l’hôte
// - Chaque auditeur écoute dans sa langue cible
// - Découpage au choix: "duration" (fixe) ou "pause" (VAD)
// =======================================================

/* ----------------------- Endpoints ----------------------- */
const API_CHUNK           = "../api/chunk.php";
const API_FEED            = "../api/feed.php";
const API_VOICES          = "../api/eleven_voices.php";
const API_LANG_TRANSLATE  = "../api/lang_translate.php";
const API_LANG_STT        = "../api/lang_stt_list.php";
const API_SETTINGS        = "../api/channel_settings.php";
const API_MUTE            = "../api/channel_mute.php";
const API_LANGS_WHITELIST = "../api/lang_whitelist.php";

/* -------------------- Constantes UI/PTT ------------------ */
// Mode de découpe: "duration" (fixe) | "pause" (VAD silence)
let sliceMode = "pause"; // bascule possible vers "duration" si besoin

// Paramètres mode "duration"
const SLICE_MS         = 2500;

// Paramètres PTT / interactions
const HOLD_DELAY_MS    = 90;
const TAP_MAX_DURATION = 160;
const DTAP_WINDOW_MS   = 300;

/* -------------------- Paramètres VAD (pause) ------------- */
const VAD = {
  threshold:     0.015, // RMS ~ [0..1] au-delà = "parole"
  silenceMs:     400,   // durée de silence pour déclencher le flush
  hangoverMs:    160,   // marge après le silence pour ne pas couper trop tôt
  minUtteranceMs:600,   // durée mini d'un segment
  maxUtteranceMs:8000,  // durée maxi d'un segment
  chunkMs:       320    // timeslice MediaRecorder (évite l'hyper-fragmentation)
};

/* ------------------------ Rôle hôte ---------------------- */
const IS_HOST_WANTED = new URL(location.href).searchParams.get("host") === "1";
let isRecognizedHost = false;

/* -------------------- État application ------------------- */
let stream;
let sessionId, clientId, channelId;
let seq = 0, sending = false;

let isJoined = false, isListening = false;
let pollTimer = null, lastSince = 0;
const playQueue = []; let isPlaying = false;

let isHolding = false, isLatched = false;
let holdTimer = null, pressStartAt = 0, lastTapAt = 0;
let holdRec = null;
let activePointerId = null;

let audioCtx = null, audioReady = false;
let allowListenerTalk = 1;
let serverMutedSpeakers = new Set();

const seenSpeakers = new Map(); // speaker_id -> { lastTs, role }

/* ----------------------- VAD Runtime --------------------- */
let analyser = null;
let analyserData = null;
let vadTimer = null;

let vadRecorder = null;
let vadChunks = [];            // fragments du segment courant (finalisé via stop)
let vadHasSpeech = false;      // indicateur parole vue récemment
let vadSegmentStartAt = 0;     // début du segment (ms epoch)
let vadLastSpeechAt = 0;       // dernier instant de parole détectée
let vadFlushPending = false;   // flush demandé (onstop n’enchaîne pas)

/* --------------------------- DOM ------------------------- */
const startBtn        = document.getElementById("startBtn");
const stopBtn         = document.getElementById("stopBtn");
const talkBtn         = document.getElementById("talkBtn");
const feedEl          = document.getElementById("feed");
const sourceLang      = document.getElementById("sourceLang");
const targetLang      = document.getElementById("targetLang");
const channelInp      = document.getElementById("channel");
const ttsPlayer       = document.getElementById("ttsPlayer");
const audioGate       = document.getElementById("audioGate");
const enableAudioBtn  = document.getElementById("enableAudioBtn");

// TTS
const ttsProviderSel  = document.getElementById("ttsProvider");
const elevenRow       = document.getElementById("elevenRow");
const elevenVoiceSel  = document.getElementById("elevenVoiceSel");
const reloadVoicesBtn = document.getElementById("reloadVoicesBtn");
const previewVoiceBtn = document.getElementById("previewVoiceBtn");
const ttsVoiceRow     = document.getElementById("ttsVoiceRow");
const ttsVoiceInput   = document.getElementById("ttsVoiceInput"); // OpenAI

// Google voix/params (optionnels)
const googleVoiceNameInp = document.getElementById("googleVoiceName");
const googleRateInp      = document.getElementById("googleRate");
const googlePitchInp     = document.getElementById("googlePitch");

// Admin hôte
const allowTalkWrap   = document.getElementById("allowTalkWrap");
const allowTalkChk    = document.getElementById("allowTalkChk");
const participantsWrap= document.getElementById("participantsWrap");
const participantsUl  = document.getElementById("participantsUl");

const elevenPreviewMap = new Map();

/* ------------------------- Utils ------------------------- */
function log(line){ feedEl.innerHTML += line + "\n"; feedEl.scrollTop = feedEl.scrollHeight; }
function ensureClientId(){
  try {
    const k='niva.clientId';
    let id = localStorage.getItem(k);
    if(!id){ id = crypto?.randomUUID?.() || String(Date.now()); localStorage.setItem(k,id); }
    clientId = id.replace(/[^a-z0-9-]/gi,'').toLowerCase();
  } catch {
    clientId = (crypto?.randomUUID?.() || String(Date.now())).replace(/[^a-z0-9-]/gi,'').toLowerCase();
  }
}
function pickMime(){
  const C = ['audio/ogg;codecs=opus','audio/webm;codecs=opus','audio/webm'];
  return C.find(m => window.MediaRecorder && MediaRecorder.isTypeSupported(m)) || '';
}
function escapeHtml(s){ return String(s).replace(/[&<>"']/g, m=>({ '&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;' }[m])); }

/* ------------------- Langues (UI) ------------------------ */
async function populateTargetLangs() {
  try {
    const res = await fetch(`${API_LANGS_WHITELIST}?t=${Date.now()}`, { cache: "no-store" });
    const data = await res.json();
    if (!data?.ok || !Array.isArray(data.target)) throw new Error("whitelist target failed");
    targetLang.innerHTML = "";
    for (const it of data.target) {
      const opt = document.createElement("option");
      opt.value = it.code;
      opt.textContent = it.name ? `${it.name} (${it.code})` : it.code;
      targetLang.appendChild(opt);
    }
    const nav = (navigator.languages && navigator.languages[0]) || navigator.language || "en-US";
    let def = Array.from(targetLang.options).find(o => o.value.toLowerCase() === String(nav).toLowerCase());
    if (!def) {
      const base = String(nav).split('-')[0].toLowerCase();
      def = Array.from(targetLang.options).find(o => o.value.split('-')[0].toLowerCase() === base);
    }
    targetLang.value = def ? def.value : (targetLang.options[0]?.value || "en-US");
  } catch {
    targetLang.innerHTML = `
      <option value="en-US">English (US)</option>
      <option value="fr-FR">Français (France)</option>
      <option value="ar">العربية</option>
      <option value="es-ES">Español (España)</option>
    `;
    const nav = (navigator.languages && navigator.languages[0]) || navigator.language || "en-US";
    targetLang.value = ["en-US","fr-FR","ar","es-ES"].find(c => c.toLowerCase() === nav.toLowerCase())
      || ["en-US","fr-FR","ar","es-ES"].find(c => c.split('-')[0].toLowerCase() === nav.split('-')[0].toLowerCase())
      || "en-US";
  }
}
async function populateSourceLangs() {
  try {
    const res = await fetch(`${API_LANGS_WHITELIST}?t=${Date.now()}`, { cache: "no-store" });
    const data = await res.json();
    if (!data?.ok || !Array.isArray(data.stt)) throw new Error("whitelist stt failed");
    sourceLang.innerHTML = "";
    for (const it of data.stt) {
      const opt = document.createElement("option");
      opt.value = it.code;
      opt.textContent = it.name ? `${it.name} — ${it.code}` : it.code;
      sourceLang.appendChild(opt);
    }
    const nav = (navigator.languages && navigator.languages[0]) || navigator.language || "fr-FR";
    let def = Array.from(sourceLang.options).find(o => o.value.toLowerCase() === String(nav).toLowerCase());
    if (!def) {
      const base = String(nav).split('-')[0].toLowerCase();
      def = Array.from(sourceLang.options).find(o => o.value.split('-')[0].toLowerCase() === base);
    }
    sourceLang.value = def ? def.value : (sourceLang.options[0]?.value || "fr-FR");
  } catch {
    sourceLang.innerHTML = `
      <option value="fr-FR">Français — fr-FR</option>
      <option value="en-US">English — en-US</option>
      <option value="ar">العربية — ar</option>
      <option value="es-ES">Español — es-ES</option>
    `;
    const nav = (navigator.languages && navigator.languages[0]) || navigator.language || "fr-FR";
    sourceLang.value = ["fr-FR","en-US","ar","es-ES"].find(c => c.toLowerCase() === nav.toLowerCase())
      || ["fr-FR","en-US","ar","es-ES"].find(c => c.split('-')[0].toLowerCase() === nav.split('-')[0].toLowerCase())
      || "fr-FR";
  }
}

/* --------------- iOS/Safari audio unlock ---------------- */
function hideAudioGate(){
  audioGate.style.display = "none";
  talkBtn.style.pointerEvents = "";
}
async function unlockAudio() {
  if (audioReady) { hideAudioGate(); return true; }
  try {
    audioCtx = audioCtx || new (window.AudioContext || window.webkitAudioContext)();
    const o = audioCtx.createOscillator();
    const g = audioCtx.createGain();
    g.gain.value = 0;
    o.connect(g); g.connect(audioCtx.destination);
    o.start(); o.stop(audioCtx.currentTime + 0.02);
    await audioCtx.resume();
    audioReady = true; hideAudioGate(); return true;
  } catch {}
  try {
    const silent = "data:audio/mp3;base64,//uQZAAAAAAAAAAAAAAAAAAAAAAAWGluZwAAAA8AAAACAAACcQCAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA";
    ttsPlayer.muted = true; ttsPlayer.src = silent;
    await ttsPlayer.play(); ttsPlayer.pause(); ttsPlayer.currentTime = 0; ttsPlayer.muted = false;
    audioReady = true; hideAudioGate(); return true;
  } catch {
    audioGate.style.display = "flex";
    talkBtn.style.pointerEvents = "none";
    return false;
  }
}
["click","touchstart"].forEach(evt => {
  window.addEventListener(evt, () => { if (!audioReady) unlockAudio(); }, { once:true, passive:true, capture:true });
});
if (enableAudioBtn) {
  enableAudioBtn.addEventListener("touchstart", (e)=>{ e.preventDefault(); }, { passive:false });
  enableAudioBtn.addEventListener("touchend",   (e)=>{ e.preventDefault(); unlockAudio(); }, { passive:false });
  enableAudioBtn.addEventListener("click",      ()=>{ unlockAudio(); });
}
["play","playing","canplay","timeupdate"].forEach(ev=>{
  ttsPlayer.addEventListener(ev, ()=>{ if(!audioReady) audioReady=true; hideAudioGate(); }, { passive:true });
});

/* -------------------- Settings & Host -------------------- */
async function getSettings() {
  if (!channelId) return;
  try {
    const url = `${API_SETTINGS}?channel=${encodeURIComponent(channelId)}&session_id=${encodeURIComponent(sessionId||'')}&speaker_id=${encodeURIComponent(clientId||'')}&t=${Date.now()}`;
    const res = await fetch(url, { cache:"no-store" });
    const data = await res.json();
    if (data?.ok && data.settings) {
      allowListenerTalk = +data.settings.allow_listener_talk ? 1 : 0;
      isRecognizedHost =
        data.settings.host_session_id &&
        data.settings.host_speaker_id &&
        String(data.settings.host_session_id).toLowerCase() === String(sessionId).toLowerCase() &&
        String(data.settings.host_speaker_id).toLowerCase() === String(clientId).toLowerCase();
      applyAllowTalkUI();
    }
  } catch {}
}
async function setSettings(newAllow) {
  if (!channelId) return;
  if (!isRecognizedHost) { log("(hôte) Vous n'êtes pas reconnu comme hôte — impossible de modifier."); return; }
  try {
    const fd = new FormData();
    fd.append("action", "set");
    fd.append("channel", channelId);
    fd.append("session_id", sessionId);
    fd.append("speaker_id", clientId);
    fd.append("allow_listener_talk", newAllow ? "1" : "0");
    const res = await fetch(`${API_SETTINGS}?t=${Date.now()}`, { method:"POST", body: fd });
    const data = await res.json();
    if (data?.ok && data.settings) {
      allowListenerTalk = +data.settings.allow_listener_talk ? 1 : 0;
      isRecognizedHost =
        data.settings.host_session_id &&
        data.settings.host_speaker_id &&
        String(data.settings.host_session_id).toLowerCase() === String(sessionId).toLowerCase() &&
        String(data.settings.host_speaker_id).toLowerCase() === String(clientId).toLowerCase();
      applyAllowTalkUI();
      log(`(hôte) Auditeurs peuvent parler : ${allowListenerTalk ? 'OUI' : 'NON'}`);
    } else {
      log("(hôte) ERREUR: impossible d'enregistrer le réglage");
    }
  } catch {
    log("(hôte) ERREUR réseau: réglage non sauvegardé");
  }
}
async function claimHost() {
  if (!IS_HOST_WANTED || !channelId) { isRecognizedHost = false; return; }
  try {
    const fd = new FormData();
    fd.append("action", "claim_host");
    fd.append("channel", channelId);
    fd.append("session_id", sessionId);
    fd.append("speaker_id", clientId);
    const res = await fetch(`${API_SETTINGS}?t=${Date.now()}`, { method:"POST", body: fd });
    const data = await res.json();
    if (data?.ok && data.settings) {
      isRecognizedHost =
        data.claimed === true &&
        data.settings.host_session_id &&
        data.settings.host_speaker_id &&
        String(data.settings.host_session_id).toLowerCase() === String(sessionId).toLowerCase() &&
        String(data.settings.host_speaker_id).toLowerCase() === String(clientId).toLowerCase();
      applyAllowTalkUI();
      if (isRecognizedHost) log("(hôte) Vous êtes l’hôte de ce channel.");
      else log("(hôte) Un autre hôte est déjà défini.");
    }
  } catch { isRecognizedHost = false; }
}
function applyAllowTalkUI() {
  if (allowTalkChk) {
    const show = isRecognizedHost === true;
    if (allowTalkWrap) allowTalkWrap.style.display = show ? "inline-flex" : "none";
    if (show) allowTalkChk.checked = (allowListenerTalk === 1);
  }
  if (!isRecognizedHost) {
    if (allowListenerTalk === 1) {
      talkBtn.classList.remove("disabled");
      talkBtn.title = "Hold-to-Talk";
      talkBtn.style.opacity = "";
    } else {
      talkBtn.classList.add("disabled");
      talkBtn.title = "Parole désactivée par l'hôte";
      talkBtn.style.opacity = "0.6";
    }
  }
}

/* ------------------------ Mutes (admin) ------------------ */
async function fetchMutes(){
  if (!channelId) return;
  try {
    const res = await fetch(`${API_MUTE}?channel=${encodeURIComponent(channelId)}&t=${Date.now()}`, { cache:"no-store" });
    const data = await res.json();
    if (data?.ok && data.mutes) {
      serverMutedSpeakers = new Set(data.mutes.speakers || []);
      renderParticipants();
    }
  } catch {}
}
async function toggleMuteSpeaker(id){
  if (!channelId || !id) return;
  const fd = new FormData();
  fd.append("channel", channelId);
  fd.append("action", "toggle");
  fd.append("type", "speaker");
  fd.append("id", id);
  try{
    const res = await fetch(`${API_MUTE}?t=${Date.now()}`, { method:"POST", body: fd });
    const data = await res.json();
    if (data?.ok && data.mutes) {
      serverMutedSpeakers = new Set(data.mutes.speakers || []);
      renderParticipants();
      log(`(hôte) ${serverMutedSpeakers.has(id) ? 'Mute' : 'Unmute'} → ${id}`);
    } else {
      log("(hôte) ERREUR: mute non sauvegardé");
    }
  }catch{ log("(hôte) ERREUR réseau: mute non sauvegardé"); }
}
function renderParticipants(){
  if (!participantsUl) return;
  participantsUl.innerHTML = "";
  const arr = Array.from(seenSpeakers.entries())
    .sort((a,b)=>b[1].lastTs - a[1].lastTs)
    .slice(0, 50);
  for (const [id, info] of arr) {
    if (id === clientId && isRecognizedHost) continue;
    const li = document.createElement("li");
    const muted = serverMutedSpeakers.has(id);
    li.innerHTML = `
      <code style="opacity:${muted?0.6:1}">${escapeHtml(id)}</code>
      <small>(${escapeHtml(info.role)})</small>
      <button data-id="${escapeHtml(id)}" class="muteBtn">${muted?'Unmute':'Mute'}</button>
    `;
    participantsUl.appendChild(li);
  }
  participantsUl.querySelectorAll(".muteBtn").forEach(btn=>{
    btn.addEventListener("click", ()=> toggleMuteSpeaker(btn.getAttribute("data-id")));
  });
}

/* ------------------------ Start/Stop --------------------- */
async function start(){
  const ch = parseFloat(channelInp.value);
  if(isNaN(ch) || ch < 0 || ch > 111.11){ log("ERREUR : channel invalide (0–111.11)"); return; }
  channelId = ch.toFixed(2);
  ensureClientId();
  sessionId = crypto?.randomUUID?.() || String(Date.now());
  seq = 0;

  await unlockAudio();

  await claimHost();
  await getSettings();
  await fetchMutes();

  isJoined = true; isListening = true;
  startPolling({ cursorNow:true });

  startBtn.disabled = true; stopBtn.disabled = false;
  talkBtn.setAttribute("aria-pressed","false");

  if (!isRecognizedHost) {
    if (ttsProviderSel) ttsProviderSel.disabled = true;
    if (elevenRow)      elevenRow.style.opacity = .6;
    if (ttsVoiceRow)    ttsVoiceRow.style.opacity = .6;
  } else {
    if (allowTalkWrap) allowTalkWrap.style.display = "inline-flex";
    if (participantsWrap) participantsWrap.style.display = "block";
  }

  log(`>> Rejoint le channel ${channelId} ${isRecognizedHost ? "(hôte)" : "(auditeur)"}`);

  if (!isRecognizedHost) {
    setInterval(getSettings, 5000);
    setInterval(fetchMutes, 7000);
  }
}
function stop(){
  isJoined = false; isListening = false;
  if(pollTimer){ clearTimeout(pollTimer); pollTimer = null; }
  lastSince = 0;

  stopLatched(); stopHold(true);
  if(stream){ try{ stream.getTracks().forEach(t=>t.stop()); }catch{} stream = null; }

  talkBtn.classList.remove('holding','latched');
  startBtn.disabled = false; stopBtn.disabled = true;
  log(">> Quitté le channel");
}

/* --------------------- Polling du feed ------------------- */
function startPolling({cursorNow=false}={}){
  const poll = async ()=>{
    if(!isListening) return;
    try{
      let url;
      const tgt = targetLang.value;
      if(cursorNow){
        url = `${API_FEED}?channel=${encodeURIComponent(channelId)}&cursor=now&limit=0&target_lang=${encodeURIComponent(tgt)}`;
        cursorNow = false;
      }else{
        url = `${API_FEED}?channel=${encodeURIComponent(channelId)}&since=${encodeURIComponent(lastSince)}&exclude_speaker_id=${encodeURIComponent(clientId)}&exclude_session_id=${encodeURIComponent(sessionId)}&viewer_is_host=${isRecognizedHost?1:0}&target_lang=${encodeURIComponent(tgt)}&limit=50`;
      }
      const res = await fetch(url);
      const data = await res.json();
      if(data?.ok){
        if (data.settings && typeof data.settings.allow_listener_talk !== "undefined") {
          allowListenerTalk = +data.settings.allow_listener_talk ? 1 : 0;
          applyAllowTalkUI();
        }
        if(typeof data.next_since === 'number') lastSince = data.next_since;
        if(Array.isArray(data.items) && data.items.length){
          for(const it of data.items){
            if(it.speaker_id && String(it.speaker_id).toLowerCase() === clientId) continue;
            if(it.session_id && String(it.session_id) === sessionId) continue;

            if (it.speaker_id) {
              seenSpeakers.set(String(it.speaker_id).toLowerCase(), { lastTs: it.ts || Date.now(), role: it.speaker_role || 'listener' });
            }

            const prov = it.tts_provider ? ` • TTS:${it.tts_provider}${it.tts_voice?`/${it.tts_voice}`:''}` : '';
            log(`[feed] ${it.channel} • ${it.source_lang}▶${it.target_lang}${prov}\nTRD: ${it.text}\n`);
            if (it.tts_url) playQueue.push(it.tts_url);
          }
          if (isRecognizedHost) renderParticipants();
          if(!isPlaying) playNext();
        }
      }
    }catch(e){ /* silencieux */ }
    finally{ pollTimer = setTimeout(poll, 900); }
  };
  if(pollTimer) clearTimeout(pollTimer);
  pollTimer = setTimeout(poll, 50);
}
if (targetLang) {
  targetLang.addEventListener('change', ()=>{
    if(!isJoined) return;
    if(pollTimer){ clearTimeout(pollTimer); pollTimer=null; }
    lastSince = 0;
    startPolling({ cursorNow:true });
  });
}

/* ---------------------- Player audio --------------------- */
async function playNext(){
  if(!isListening || isPlaying) return;
  const next = playQueue.shift(); if(!next) return;

  if (!audioReady) {
    const ok = await unlockAudio();
    if (!ok) { playQueue.unshift(next); setTimeout(playNext, 800); return; }
  }

  isPlaying = true;
  ttsPlayer.src = next;
  try { await ttsPlayer.play(); }
  catch {
    if (!audioReady) { audioGate.style.display = "flex"; talkBtn.style.pointerEvents = "none"; }
    playQueue.unshift(next);
    isPlaying = false;
    return;
  }
  const onDone = ()=>{ ttsPlayer.removeEventListener('ended', onDone); ttsPlayer.removeEventListener('error', onErr); isPlaying=false; playNext(); };
  const onErr  = ()=>{ ttsPlayer.removeEventListener('ended', onDone); ttsPlayer.removeEventListener('error', onErr); isPlaying=false; playNext(); };
  ttsPlayer.addEventListener('ended', onDone, { once:true });
  ttsPlayer.addEventListener('error', onErr, { once:true });
}

/* -------------------------- Mic -------------------------- */
async function ensureMic(){
  if(stream) return stream;
  stream = await navigator.mediaDevices.getUserMedia({
    audio:{ channelCount:1, sampleRate:48000, echoCancellation:true, noiseSuppression:false, autoGainControl:false }
  });

  // Prépare l'analyser pour VAD si nécessaire
  if (!audioCtx) {
    audioCtx = new (window.AudioContext || window.webkitAudioContext)();
  }
  const srcNode = audioCtx.createMediaStreamSource(stream);
  analyser = audioCtx.createAnalyser();
  analyser.fftSize = 1024;
  analyserData = new Uint8Array(analyser.fftSize);
  srcNode.connect(analyser);

  return stream;
}

/* ------------------------ VAD helpers -------------------- */
function getRms() {
  if (!analyser || !analyserData) return 0;
  analyser.getByteTimeDomainData(analyserData);
  let sum = 0;
  for (let i=0;i<analyserData.length;i++){
    const v = (analyserData[i] - 128) / 128;
    sum += v*v;
  }
  return Math.sqrt(sum / analyserData.length);
}

function startVadRecorder() {
  if (!stream) return;
  const mime = pickMime();
  vadChunks = [];
  const rec = new MediaRecorder(stream, { mimeType: mime, audioBitsPerSecond: 128000 });
  vadRecorder = rec;

  rec.ondataavailable = (e)=>{
    if (e.data && e.data.size) {
      vadChunks.push(e.data);
      if (!vadSegmentStartAt) vadSegmentStartAt = Date.now();
    }
  };
  rec.onstop = async ()=>{
    try {
      if (!vadChunks.length) return;
      const mimeType = pickMime().includes('ogg') ? 'audio/ogg' : 'audio/webm';
      const blob = new Blob(vadChunks, { type: mimeType });
      vadChunks = [];
      await sendChunk(blob);
    } finally {
      if ((isHolding || isLatched) && !document.hidden && !vadFlushPending) {
        // Nouveau segment
        vadSegmentStartAt = 0;
        vadLastSpeechAt = 0;
        vadHasSpeech = false;
        startVadRecorder();
      }
      if (vadFlushPending) {
        // Flush demandé (fin d’appui) → reset complet
        vadFlushPending = false;
        vadSegmentStartAt = 0;
        vadLastSpeechAt = 0;
        vadHasSpeech = false;
      }
    }
  };

  // On obtient des micro-chunks réguliers mais on laisse stop() finaliser le conteneur
  rec.start(VAD.chunkMs);
}

function vadStart() {
  vadChunks = [];
  vadHasSpeech = false;
  vadSegmentStartAt = 0;
  vadLastSpeechAt = 0;
  vadFlushPending = false;
  if (vadTimer) clearInterval(vadTimer);
  vadTimer = setInterval(vadTick, 80);
  startVadRecorder();
}

function vadStop(flushRemaining) {
  if (vadTimer) { clearInterval(vadTimer); vadTimer = null; }
  if (flushRemaining) {
    if (vadRecorder && vadRecorder.state !== 'inactive') {
      vadFlushPending = true; // on laisse onstop envoyer le dernier segment
      try { vadRecorder.stop(); } catch {}
      return;
    }
  }
  // reset immédiat si pas de flush
  vadChunks = [];
  vadHasSpeech = false;
  vadSegmentStartAt = 0;
  vadLastSpeechAt = 0;
  vadFlushPending = false;
}

function vadTick() {
  if (!vadRecorder || vadRecorder.state === 'inactive') return;

  const now = Date.now();
  const vol = getRms();
  if (vol > VAD.threshold) {
    vadHasSpeech = true;
    vadLastSpeechAt = now;
  }
  const elapsed = vadSegmentStartAt ? (now - vadSegmentStartAt) : 0;
  const sinceLastTalk = vadLastSpeechAt ? (now - vadLastSpeechAt) : elapsed;

  // Conditions de fin de segment
  const hadMin = elapsed >= VAD.minUtteranceMs;
  const longPause = sinceLastTalk >= (VAD.silenceMs + VAD.hangoverMs);
  const tooLong = elapsed >= VAD.maxUtteranceMs;

  if ((vadHasSpeech && hadMin && longPause) || tooLong) {
    // Finalise proprement ce segment (Blob valide) → onstop fera l’envoi
    vadFlushPending = false;
    try { vadRecorder.stop(); } catch {}
  }
}

/* --------------------- PTT : Hold / Latch ---------------- */
// HOLD: appui maintenu
async function startHold(){
  if(!isJoined || isHolding || isLatched) return;
  if (!isRecognizedHost && allowListenerTalk !== 1) { log("(Parole des auditeurs désactivée)"); return; }
  if (serverMutedSpeakers.has(clientId.toLowerCase())) { log("(Vous êtes mute par l'hôte)"); return; }

  await ensureMic();
  const mime = pickMime();
  isHolding = true;
  talkBtn.classList.add('holding');
  talkBtn.setAttribute("aria-pressed","true");

  if (sliceMode === "pause") {
    // Mode VAD: analyse + recorder cyclique (finalisation via stop)
    vadStart();
  } else {
    // Mode durée: un seul blob à la fin
    holdRec = { mime, chunks:[], recorder:null };
    const rec = new MediaRecorder(stream, { mimeType: mime, audioBitsPerSecond: 128000 });
    holdRec.recorder = rec;
    rec.ondataavailable = (e)=>{ if(e.data && e.data.size) holdRec.chunks.push(e.data); };
    rec.onstop = async ()=>{
      try{
        if(!holdRec) return;
        const blob = new Blob(holdRec.chunks, { type: holdRec.mime || 'audio/webm' });
        holdRec = null;
        await sendChunk(blob);
      }finally{ isHolding = false; }
    };
    rec.start();
  }
}
function stopHold(){
  if (sliceMode === "pause") {
    talkBtn.classList.remove('holding');
    talkBtn.setAttribute("aria-pressed","false");
    // Demande un flush final via onstop (blob propre)
    try { if (vadRecorder && vadRecorder.state !== 'inactive') { vadFlushPending = true; vadRecorder.stop(); } } catch {}
    vadRecorder = null;
    vadStop(false); // onstop a déjà l’envoi
    isHolding = false;
    return;
  }
  if(!holdRec?.recorder) return;
  talkBtn.classList.remove('holding');
  talkBtn.setAttribute("aria-pressed","false");
  try{ if(holdRec.recorder.state !== 'inactive') holdRec.recorder.stop(); }catch{}
}

// LATCHED: double tap -> envoi continu selon sliceMode
async function startLatched(){
  if(!isJoined || isLatched) return;
  if (!isRecognizedHost && allowListenerTalk !== 1) { log("(Parole des auditeurs désactivée)"); return; }
  if (serverMutedSpeakers.has(clientId.toLowerCase())) { log("(Vous êtes mute par l'hôte)"); return; }

  await ensureMic();
  isLatched = true;
  talkBtn.classList.add('latched');
  talkBtn.setAttribute("aria-pressed","true");

  if (sliceMode === "pause") {
    vadStart();
  } else {
    const loop = ()=>{
      if(!isLatched) return;
      const mime = pickMime();
      const rec = new MediaRecorder(stream,{ mimeType:mime, audioBitsPerSecond:128000 });
      rec.ondataavailable = (e)=>{ if(e.data && e.data.size) sendChunk(e.data); };
      rec.onstop = ()=>{ if(isLatched) setTimeout(loop,0); };
      rec.start();
      setTimeout(()=>{ try{ if(rec.state!=='inactive') rec.stop(); }catch{} }, SLICE_MS);
    };
    loop();
  }
}
function stopLatched(){
  if(!isLatched) return;
  isLatched = false;
  talkBtn.classList.remove('latched');
  talkBtn.setAttribute("aria-pressed","false");
  if (sliceMode === "pause") {
    try { if(vadRecorder && vadRecorder.state !== 'inactive') { vadFlushPending = true; vadRecorder.stop(); } } catch {}
    vadRecorder = null;
    vadStop(false);
  }
}

/* ------------------------ Envoi chunk -------------------- */
async function sendChunk(blob){
  while(sending){ await new Promise(r=>setTimeout(r,40)); }
  sending = true;
  try{
    const fd = new FormData();
    const ext = blob.type.includes('ogg') ? 'ogg' : 'webm';
    fd.append("audio", blob, `seq-${seq}.${ext}`);
    fd.append("session_id", sessionId);
    fd.append("speaker_id", clientId);
    fd.append("speaker_role", isRecognizedHost ? "host" : "listener");
    fd.append("channel", channelId);
    fd.append("seq", String(seq));
    fd.append("source_lang", sourceLang.value);

    const provider = ttsProviderSel ? ttsProviderSel.value : '';
    if (isRecognizedHost && provider) {
      fd.append("tts_provider", provider);

      if (provider === 'elevenlabs' && elevenVoiceSel) {
        fd.append("tts_voice", elevenVoiceSel.value || '');
      } else if (provider === 'openai' && ttsVoiceInput) {
        fd.append("tts_voice", (ttsVoiceInput.value || '').trim());
      } else if (provider === 'google') {
        const name  = (googleVoiceNameInp?.value || "").trim();
        const rate  = parseFloat(googleRateInp?.value ?? "1.0");
        const pitch = parseFloat(googlePitchInp?.value ?? "0");
        fd.append("tts_voice", name || "tier:neural2");
        fd.append("tts_rate",  String(isFinite(rate)  ? rate  : 1.0));
        fd.append("tts_pitch", String(isFinite(pitch) ? pitch : 0.0));
      } else {
        fd.append("tts_voice", "");
      }
    } else {
      fd.append("tts_provider", "");
      fd.append("tts_voice", "");
    }

    const res = await fetch(API_CHUNK, { method:"POST", body:fd });
    const raw = await res.text();
    let json; try{ json = JSON.parse(raw); }catch{ log(`ERREUR serveur (non-JSON): ${raw.slice(0,200)}...`); return; }
    const seqLabel = (json && json.seq !== null && typeof json.seq !== 'undefined') ? json.seq : '?';
    if(json.ok){
      const pv = json.tts_provider ? ` • TTS:${json.tts_provider}${json.tts_voice?`/${json.tts_voice}`:''}` : '';
      log(`[${seqLabel}] ${json.source_lang} ▶ (multi)\nSTT: ${json.text}${pv?`\nCHOIX HÔTE:${pv}`:''}\n`);
    }else{
      log(`ERREUR [${seqLabel}] : ${json.error || "inconnue"}`);
    }
  }catch(e){
    log("ERREUR réseau : " + (e?.message || e));
  }finally{
    seq++; sending=false;
  }
}

/* ----------------- Voix ElevenLabs (UI) ----------------- */
async function loadElevenVoices(refresh=false) {
  if (!elevenVoiceSel) return;
  try {
    elevenVoiceSel.innerHTML = `<option value="">Chargement…</option>`;
    const url = refresh ? `${API_VOICES}?refresh=1` : API_VOICES;
    const res = await fetch(url);
    const data = await res.json();
    if (!data?.ok || !Array.isArray(data.voices)) {
      elevenVoiceSel.innerHTML = `<option value="">Erreur chargement</option>`;
      return;
    }
    elevenPreviewMap.clear();
    const opts = [`<option value="">— Choisir une voix —</option>`];
    for (const v of data.voices) {
      const label = [v.name, (v.labels?.accent||''), (v.labels?.gender||''), (v.labels?.age||'')].filter(Boolean).join(' ');
      const preview = v.preview_url || '';
      elevenPreviewMap.set(v.id, preview);
      opts.push(`<option value="${escapeHtml(v.id)}" data-preview="${escapeHtml(preview)}">${escapeHtml(label)}</option>`);
    }
    elevenVoiceSel.innerHTML = opts.join('');
  } catch {
    elevenVoiceSel.innerHTML = `<option value="">Erreur réseau</option>`;
  }
}
if (reloadVoicesBtn) reloadVoicesBtn.addEventListener('click', ()=> loadElevenVoices(true));
if (previewVoiceBtn) previewVoiceBtn.addEventListener('click', async ()=>{
  if (!elevenVoiceSel) return;
  const id = elevenVoiceSel.value;
  if (!id) return;
  const url = elevenPreviewMap.get(id);
  if (!url) return;
  await unlockAudio();
  try { ttsPlayer.src = url; await ttsPlayer.play(); } catch {}
});

/* --------------- Provider ↔ champs voix UI --------------- */
function syncTtsUi() {
  if (!ttsProviderSel) return;
  const p = ttsProviderSel.value;
  if (elevenRow)   elevenRow.style.display = (p === 'elevenlabs') ? 'flex' : 'none';
  if (ttsVoiceRow) ttsVoiceRow.style.display = (p === 'openai')    ? 'flex' : 'none';
  if (p === 'elevenlabs' && elevenVoiceSel && (!elevenVoiceSel.options.length || elevenVoiceSel.value === '')) {
    loadElevenVoices(false);
  }
}
if (ttsProviderSel) ttsProviderSel.addEventListener('change', syncTtsUi);

/* ------------------- PTT : gestion UI -------------------- */
function onPointerDown(ev){
  if(!isJoined) return;
  if (!isRecognizedHost && allowListenerTalk !== 1) return;
  if (serverMutedSpeakers.has(clientId.toLowerCase())) { log("(Vous êtes mute par l'hôte)"); return; }

  ev.preventDefault();
  unlockAudio();
  activePointerId = ev.pointerId;
  try{ talkBtn.setPointerCapture(ev.pointerId); }catch{}
  pressStartAt = Date.now();
  holdTimer = setTimeout(()=>{ startHold(); }, HOLD_DELAY_MS);
}
function onPointerUp(ev){
  if(!isJoined) return;
  ev.preventDefault();
  if(activePointerId !== null){ try{ talkBtn.releasePointerCapture(activePointerId); }catch{} activePointerId = null; }

  const now = Date.now();
  const pressDur = now - pressStartAt;
  if(holdTimer){ clearTimeout(holdTimer); holdTimer = null; }

  if(pressDur <= TAP_MAX_DURATION){
    if(now - lastTapAt <= DTAP_WINDOW_MS){
      if(isLatched) stopLatched(); else startLatched();
      lastTapAt = 0;
    }else{
      lastTapAt = now;
    }
  }else{
    if(!isLatched) stopHold();
  }
}
function onPointerCancel(){
  if(holdTimer){ clearTimeout(holdTimer); holdTimer = null; }
  if(!isLatched) stopHold();
  activePointerId = null;
}

/* ---------------------- DOM bindings -------------------- */
document.addEventListener('DOMContentLoaded', async ()=>{
  if (channelInp && !channelInp.value) channelInp.value = '1.01';
  await populateTargetLangs();
  await populateSourceLangs();
  syncTtsUi();

  if (allowTalkChk) {
    allowTalkWrap && (allowTalkWrap.style.display = "none"); // affiché après claim si hôte
    allowTalkChk.addEventListener('change', ()=> setSettings(allowTalkChk.checked ? 1 : 0));
  }
  if (participantsWrap) participantsWrap.style.display = "none";
});
if (startBtn) startBtn.addEventListener("click", start);
if (stopBtn)  stopBtn .addEventListener("click", stop);

if (talkBtn) {
  talkBtn.addEventListener("pointerdown", onPointerDown);
  talkBtn.addEventListener("pointerup", onPointerUp);
  talkBtn.addEventListener("pointercancel", onPointerCancel);
  talkBtn.addEventListener("lostpointercapture", onPointerCancel);
  talkBtn.addEventListener("pointerleave", (e)=>{ if(e.buttons===0) onPointerCancel(); });
}

window.addEventListener("visibilitychange", ()=>{ if(document.hidden){ if(!isLatched) stopHold(); }});
window.addEventListener("beforeunload", ()=>{ try{ stop(); }catch{} });
