<?php
// api/feed.php — par auditeur : traduit vers sa langue, synthétise avec le provider/voix choisis par l’hôte
declare(strict_types=1);

ini_set('display_errors', '1');
error_reporting(E_ALL);
header('Content-Type: application/json; charset=utf-8');

require_once dirname(__DIR__) . '/vendor/autoload.php';
require_once __DIR__ . '/config.php';

use Google\Cloud\Translate\V2\TranslateClient;

function out(int $code, array $payload): never {
  http_response_code($code);
  echo json_encode($payload, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
  exit;
}
function sanitize_channel(string $raw): array {
  $raw = trim($raw);
  if ($raw === '' || !is_numeric($raw)) return [null, null, 'channel invalide'];
  $f = (float)$raw;
  if ($f < 0 || $f > 111.11) return [null, null, 'channel hors limites'];
  $norm = number_format($f, 2, '.', '');
  $file = str_replace('.', '_', $norm);
  return [$norm, $file, null];
}
function http_post_json(string $url, array $headers, array $jsonBody): array {
  $ch = curl_init($url);
  $hdrs = []; foreach ($headers as $k=>$v) $hdrs[] = $k.': '.$v;
  curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_POST => true,
    CURLOPT_HTTPHEADER => $hdrs,
    CURLOPT_POSTFIELDS => json_encode($jsonBody, JSON_UNESCAPED_UNICODE),
    CURLOPT_HEADER => false,
  ]);
  $body = curl_exec($ch);
  $err  = curl_error($ch);
  $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
  curl_close($ch);
  return ['ok'=>($err==='' && $code>=200 && $code<300), 'code'=>$code, 'err'=>$err, 'body'=>$body];
}
function tts_cached_path(string $sessionId, string $eventId, string $tgt, string $provider): string {
  @mkdir(TTS_PUBLIC_DIR . '/' . $sessionId, 0775, true);
  $safeProv = preg_replace('/[^a-z0-9_-]/i', '', $provider);
  $safeTgt  = preg_replace('/[^a-z0-9_-]/i', '', $tgt);
  return TTS_PUBLIC_DIR . '/' . $sessionId . '/' . $eventId . '-' . $safeTgt . '-' . $safeProv . '.mp3';
}
function tts_public_url(string $sessionId, string $eventId, string $tgt, string $provider): string {
  $safeProv = preg_replace('/[^a-z0-9_-]/i', '', $provider);
  $safeTgt  = preg_replace('/[^a-z0-9_-]/i', '', $tgt);
  return 'tts/' . $sessionId . '/' . $eventId . '-' . $safeTgt . '-' . $safeProv . '.mp3';
}
function translate_text(string $text, string $target): string {
  $gcpCred = getenv('GOOGLE_APPLICATION_CREDENTIALS') ?: (defined('GCP_CREDENTIALS') ? GCP_CREDENTIALS : null);
  $tr = new TranslateClient(['keyFilePath' => $gcpCred]);
  $res = $tr->translate($text, ['target' => $target, 'format'=>'text']);
  return $res['text'] ?? $text;
}

/* ---------------- Google TTS : listVoices + choix + synthèse ---------------- */

// Cache 24h de la liste des voix (évite l’appel à chaque requête)
function load_google_voices(): array {
  $cacheDir = __DIR__ . '/cache'; @mkdir($cacheDir, 0775, true);
  $cacheFile = $cacheDir . '/google_voices.json';
  if (is_file($cacheFile) && (time() - filemtime($cacheFile) < 86400)) {
    $j = json_decode((string)@file_get_contents($cacheFile), true);
    if (is_array($j)) return $j;
  }
  $gcp = getenv('GOOGLE_APPLICATION_CREDENTIALS') ?: (defined('GCP_CREDENTIALS') ? GCP_CREDENTIALS : null);
  $tts = new Google\Cloud\TextToSpeech\V1\TextToSpeechClient(['transport'=>'rest','credentials'=>$gcp]);
  $resp = $tts->listVoices();
  $voices = [];
  foreach ($resp->getVoices() as $v) {
    $voices[] = [
      'name' => $v->getName(), // ex: en-US-Neural2-F
      'lang' => iterator_to_array($v->getLanguageCodes()),
      'gender' => $v->getSsmlGender(),
      'rate_hz' => $v->getNaturalSampleRateHertz(),
    ];
  }
  @file_put_contents($cacheFile, json_encode($voices, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES));
  return $voices;
}

// Sélection d'une voix adaptée à la langue cible (supporte "tier:neural2|wavenet|standard" ou nom exact)
function choose_google_voice(string $target, ?string $preferTierOrName): array {
  $voices = load_google_voices();

  // 1) si on a un nom exact
  if ($preferTierOrName && !str_starts_with(strtolower($preferTierOrName), 'tier:') && strtolower($preferTierOrName) !== 'auto') {
    foreach ($voices as $v) if (strcasecmp($v['name'], $preferTierOrName) === 0) return $v;
  }

  // 2) map langue ISO -> BCP47 utilisé par Google TTS
  $map = [
    'en'=>'en-US','fr'=>'fr-FR','ar'=>'ar-XA','es'=>'es-ES','pt'=>'pt-BR','de'=>'de-DE','it'=>'it-IT','nl'=>'nl-NL',
    'ru'=>'ru-RU','tr'=>'tr-TR','ja'=>'ja-JP','ko'=>'ko-KR','zh'=>'zh-CN','zh-cn'=>'zh-CN','zh-tw'=>'zh-TW','zh-hk'=>'zh-HK'
  ];
  $lang = strtolower($target);
  $want = $map[$lang] ?? $target;

  // 3) ordre de préférence des tiers
  $tiersPref = ['neural2','wavenet','standard'];
  if ($preferTierOrName && str_starts_with(strtolower($preferTierOrName), 'tier:')) {
    $t = strtolower(substr($preferTierOrName, 5));
    $tiersPref = array_values(array_unique(array_merge([$t], $tiersPref)));
  }

  $tierOf = function(string $name): string {
    $n = strtolower($name);
    if (str_contains($n,'neural2')) return 'neural2';
    if (str_contains($n,'wavenet')) return 'wavenet';
    return 'standard';
  };

  // 4) candidats par langue + tri par tier + nom
  $cands = array_values(array_filter($voices, fn($v) => array_reduce($v['lang'], fn($carry,$lc)=>$carry || stripos($lc, $want)===0, false)));
  usort($cands, function($a,$b) use($tiersPref,$tierOf){
    $ia = array_search($tierOf($a['name']), $tiersPref, true);
    $ib = array_search($tierOf($b['name']), $tiersPref, true);
    return ($ia <=> $ib) ?: strcmp($a['name'],$b['name']);
  });

  return $cands[0] ?? ['name'=>null,'lang'=>[$want],'gender'=>null,'rate_hz'=>null];
}

// Synthèse Google paramétrable (voix nominative OU "tier:neural2", rate/pitch/effects)
function synth_google(string $text, string $target, string $outPath, ?string $voiceName=null, float $rate=1.0, float $pitch=0.0, array $effects=[]): void {
  $gcp = getenv('GOOGLE_APPLICATION_CREDENTIALS') ?: (defined('GCP_CREDENTIALS') ? GCP_CREDENTIALS : null);

  $pick = choose_google_voice($target, $voiceName ?: 'tier:neural2');
  $voiceLang = $pick['lang'][0] ?? $target;
  $voiceName = $pick['name'] ?? null;

  $ttsClient = new Google\Cloud\TextToSpeech\V1\TextToSpeechClient(['transport'=>'rest','credentials'=>$gcp]);

  $inputText = (new Google\Cloud\TextToSpeech\V1\SynthesisInput())->setText($text);

  $voiceSel  = new Google\Cloud\TextToSpeech\V1\VoiceSelectionParams();
  $voiceSel->setLanguageCode($voiceLang);
  if ($voiceName) $voiceSel->setName($voiceName);

  $audioCfg  = new Google\Cloud\TextToSpeech\V1\AudioConfig();
  $audioCfg->setAudioEncoding(Google\Cloud\TextToSpeech\V1\AudioEncoding::MP3);
  if ($rate  > 0)  $audioCfg->setSpeakingRate($rate);  // 0.25–4.0
  $audioCfg->setPitch($pitch);                          // -20..+20
  if (!empty($effects)) $audioCfg->setEffectsProfileId($effects);

  $ttsResp   = $ttsClient->synthesizeSpeech($inputText, $voiceSel, $audioCfg);
  $audio     = $ttsResp->getAudioContent();
  if (method_exists($ttsClient, 'close')) $ttsClient->close();

  file_put_contents($outPath, $audio);
}

/* --------------------------- Eleven / OpenAI TTS --------------------------- */

function synth_eleven(string $text, string $voiceId, string $outPath): bool {
  $api = defined('ELEVEN_API_KEY') ? ELEVEN_API_KEY : '';
  if (!$api) return false;
  $url = 'https://api.elevenlabs.io/v1/text-to-speech/'.rawurlencode($voiceId ?: '21m00Tcm4TlvDq8ikWAM');
  $res = http_post_json($url, [
    'accept' => 'audio/mpeg',
    'xi-api-key' => $api,
    'Content-Type'=> 'application/json'
  ], [
    'text'=>$text, 'model_id'=>'eleven_multilingual_v2',
    'voice_settings'=>['stability'=>0.5,'similarity_boost'=>0.75],
    'output_format'=>'mp3_44100_128'
  ]);
  if (!$res['ok'] || !is_string($res['body']) || strlen($res['body'])===0) return false;
  file_put_contents($outPath, $res['body']);
  return true;
}
function synth_openai(string $text, string $voice, string $outPath): bool {
  $api = defined('OPENAI_API_KEY') ? OPENAI_API_KEY : '';
  if (!$api) return false;
  $v = strtolower(trim($voice ?: 'alloy'));
  $map = ['allow'=>'alloy','aloy'=>'alloy','versa'=>'verse','vèrse'=>'verse']; if (isset($map[$v])) $v = $map[$v];
  $res = http_post_json('https://api.openai.com/v1/audio/speech', [
    'Authorization'=>'Bearer '.$api,'Accept'=>'audio/mpeg','Content-Type'=>'application/json'
  ], [
    'model'=>'gpt-4o-mini-tts','input'=>$text,'voice'=>$v,'format'=>'mp3'
  ]);
  if (!$res['ok'] || !is_string($res['body']) || strlen($res['body'])===0) return false;
  file_put_contents($outPath, $res['body']);
  return true;
}

/* --------------------------- Entrée requête --------------------------- */
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

$channelRaw    = $_GET['channel'] ?? null;
$since         = isset($_GET['since']) ? (float)$_GET['since'] : null;
$cursorNow     = isset($_GET['cursor']) && $_GET['cursor']==='now';
$limit         = max(0, (int)($_GET['limit'] ?? 50));
$excludeSpk    = strtolower(trim($_GET['exclude_speaker_id'] ?? ''));
$excludeSess   = strtolower(trim($_GET['exclude_session_id'] ?? ''));
$viewerIsHost  = (int)($_GET['viewer_is_host'] ?? 0) === 1;
$targetLang    = strtolower(trim($_GET['target_lang'] ?? 'en')); // ← langue de l’AUDITEUR

if (!$channelRaw) out(400, ['ok'=>false,'error'=>'channel requis']);
[$channelNorm, $channelFile, $chanErr] = sanitize_channel($channelRaw);
if ($chanErr !== null) out(400, ['ok'=>false,'error'=>'channel invalide (0..111.11)']);

$feedPath = FEED_DIR . '/' . $channelFile . '.ndjson';
if (!is_file($feedPath)) out(200, ['ok'=>true, 'items'=>[], 'next_since'=>($since ?: microtime(true))]);

$lines = @file($feedPath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
if ($lines === false) out(500, ['ok'=>false, 'error'=>'Lecture du feed échouée']);

$now = microtime(true);
$startTs = $cursorNow ? $now : ($since ?: 0.0);

$items = [];
$maxTs = $startTs;
foreach ($lines as $line) {
  $ev = json_decode($line, true);
  if (!$ev || !isset($ev['ts'])) continue;

  if ($ev['ts'] <= $startTs) continue;
  if ($viewerIsHost && ($ev['speaker_role'] ?? '') === 'listener') continue;
  if ($excludeSpk && strtolower($ev['speaker_id'] ?? '') === $excludeSpk) continue;
  if ($excludeSess && strtolower($ev['session_id'] ?? '') === $excludeSess) continue;

  $maxTs = max($maxTs, (float)$ev['ts']);

  $srcText = trim((string)($ev['stt'] ?? ''));
  if ($srcText === '') continue;

  // 1) Traduction vers la langue demandée par l’auditeur
  $translated = translate_text($srcText, $targetLang);

  // 2) TTS selon le provider/voix/réglages choisis par l’hôte dans l’évènement
  $provider = strtolower($ev['tts_provider'] ?? (defined('TTS_DEFAULT_PROVIDER') ? TTS_DEFAULT_PROVIDER : 'google'));
  $voice    = $ev['tts_voice'] ?? null;
  $rate     = isset($ev['tts_rate'])  ? (float)$ev['tts_rate']  : 1.0;
  $pitch    = isset($ev['tts_pitch']) ? (float)$ev['tts_pitch'] : 0.0;

  $mp3Path  = tts_cached_path((string)$ev['session_id'], (string)$ev['id'], $targetLang, $provider);
  $mp3Url   = tts_public_url((string)$ev['session_id'], (string)$ev['id'], $targetLang, $provider);

  if (!is_file($mp3Path)) {
    $ok = false;
    if ($provider === 'elevenlabs') $ok = synth_eleven($translated, (string)$voice, $mp3Path);
    else if ($provider === 'openai') $ok = synth_openai($translated, (string)$voice, $mp3Path);

    if (!$ok) { // Google ou fallback Google avec sélection de voix + réglages
      synth_google($translated, $targetLang, $mp3Path, $voice ?: 'tier:neural2', $rate, $pitch, /*effects*/[]);
      $provider = 'google';
      // $voice (nom exact) peut être différent par langue -> pas nécessaire dans la réponse
    }
  }

  $items[] = [
    'ts'            => $ev['ts'],
    'id'            => $ev['id'],
    'session_id'    => $ev['session_id'],
    'channel'       => $ev['channel'],
    'speaker_id'    => $ev['speaker_id'],
    'speaker_role'  => $ev['speaker_role'],
    'source_lang'   => $ev['source_lang'],
    'target_lang'   => $targetLang,
    'text'          => $translated,
    'tts_url'       => $mp3Url,
    'tts_provider'  => $provider,
    'tts_voice'     => $voice
  ];

  if ($limit > 0 && count($items) >= $limit) break;
}

out(200, ['ok'=>true,'items'=>$items,'next_since'=>$maxTs > $startTs ? $maxTs : $now]);
