<?php
// api/config.php
declare(strict_types=1);

/**
 * Config unifiée : chemins + CORS + limites + fallbacks GCP + TTS + whitelist langues
 * - Garantit GOOGLE_APPLICATION_CREDENTIALS & GOOGLE_CLOUD_PROJECT côté PHP-FPM
 * - Préserve la whitelist et les fonctions de normalisation STT / cible
 */

ini_set('display_errors', '1');
error_reporting(E_ALL);

// -------------------------------
// Chemins / Dossiers de travail
// -------------------------------
define('ROOT_DIR', dirname(__DIR__));
define('API_DIR', __DIR__);
define('PUBLIC_DIR', ROOT_DIR . '/public');

define('SESSIONS_DIR', API_DIR . '/sessions');   // uploads audio (chunks)
define('TTS_PUBLIC_DIR', PUBLIC_DIR . '/tts');   // sorties MP3 jouables
define('FEED_DIR', API_DIR . '/feeds');          // bus NDJSON par channel
define('CACHE_DIR', API_DIR . '/cache');         // petits caches (voix, etc.)

@mkdir(SESSIONS_DIR, 0775, true);
@mkdir(TTS_PUBLIC_DIR, 0775, true);
@mkdir(FEED_DIR, 0775, true);
@mkdir(CACHE_DIR, 0775, true);

// -------------------------------
/* CORS + limites HTTP utiles */
// -------------------------------
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Headers: Content-Type');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');

@ini_set('upload_max_filesize', '50M');
@ini_set('post_max_size', '50M');
@ini_set('max_execution_time', '120');
@ini_set('memory_limit', '256M');

// -------------------------------
// Fallbacks Google ADC (clé + project)
// -------------------------------
// Important : la lib Google lit les variables d'environnement (ENV), pas une constante.
$cred = getenv('GOOGLE_APPLICATION_CREDENTIALS');

if (!$cred || !is_readable($cred)) {
  // <<< Remplace par TON chemin absolu si différent >>>
  $cred = '/home/miva/FileDocs/miva-connect-473519-75b4e78ab97f.json';
  if (is_readable($cred)) {
    putenv('GOOGLE_APPLICATION_CREDENTIALS=' . $cred);
    $_ENV['GOOGLE_APPLICATION_CREDENTIALS']  = $cred;
    $_SERVER['GOOGLE_APPLICATION_CREDENTIALS'] = $cred;
  }
}
define('GCP_CREDENTIALS', $cred ?: ''); // utile pour debug interne éventuel

// Assurer le project_id via ENV si absent (ou l'extraire du JSON)
if (!getenv('GOOGLE_CLOUD_PROJECT') && !getenv('GCLOUD_PROJECT')) {
  $projectId = null;
  if ($cred && is_readable($cred)) {
    $j = json_decode((string)@file_get_contents($cred), true);
    $projectId = $j['project_id'] ?? null;
  }
  if (!$projectId) {
    // <<< fallback explicite sûr si le JSON n'est pas lisible côté FPM >>>
    $projectId = 'miva-connect-473519';
  }
  putenv('GOOGLE_CLOUD_PROJECT=' . $projectId);
  putenv('GCLOUD_PROJECT=' . $projectId);
  $_ENV['GOOGLE_CLOUD_PROJECT']   = $projectId;
  $_ENV['GCLOUD_PROJECT']         = $projectId;
  $_SERVER['GOOGLE_CLOUD_PROJECT']= $projectId;
  $_SERVER['GCLOUD_PROJECT']      = $projectId;
}

// -------------------------------
// Providers TTS (choix & clés)
// -------------------------------
define('TTS_DEFAULT_PROVIDER', strtolower(getenv('TTS_DEFAULT_PROVIDER') ?: 'google'));

$__eleven = getenv('ELEVENLABS_API_KEY') ?: getenv('ELEVEN_API_KEY')
          ?: ($_SERVER['ELEVENLABS_API_KEY'] ?? $_SERVER['ELEVEN_API_KEY'] ?? null)
          ?: ($_SERVER['REDIRECT_ELEVENLABS_API_KEY'] ?? $_SERVER['REDIRECT_ELEVEN_API_KEY'] ?? '');
define('ELEVEN_API_KEY', $__eleven ?: '');

$__openai = getenv('OPENAI_API_KEY')
          ?: ($_SERVER['OPENAI_API_KEY'] ?? null)
          ?: ($_SERVER['REDIRECT_OPENAI_API_KEY'] ?? '');
define('OPENAI_API_KEY', $__openai ?: '');

// ========================
//  WHITELIST DE LANGUES (STT/Target au niveau serveur)
// ========================
// Chaque entrée : code => [ 'name' => label lisible, 'stt' => bool, 'target' => bool ]
// - 'stt' = peut être utilisée comme langue source (Speech-to-Text)
// - 'target' = peut être utilisée comme langue cible d’écoute / diffusion (Traduction + TTS)
const LANG_WHITELIST = [
  'fr-FR' => ['name' => 'Français (France)',  'stt' => true, 'target' => true],
  'en-US' => ['name' => 'English (US)',       'stt' => true, 'target' => true],
  'en-GB' => ['name' => 'English (UK)',       'stt' => false,'target' => true], // STT mappe vers en-US
  'ar'    => ['name' => 'العربية',           'stt' => true, 'target' => true],  // v2 accepte 'ar' générique
  'es-ES' => ['name' => 'Español (España)',   'stt' => true, 'target' => true],
  'pt-BR' => ['name' => 'Português (Brasil)', 'stt' => true, 'target' => true],
  'de-DE' => ['name' => 'Deutsch (DE)',       'stt' => true, 'target' => true],
  'it-IT' => ['name' => 'Italiano (IT)',      'stt' => true, 'target' => true],
  'nl-NL' => ['name' => 'Nederlands (NL)',    'stt' => true, 'target' => true],
  'tr-TR' => ['name' => 'Türkçe (TR)',        'stt' => true, 'target' => true],
  'ru-RU' => ['name' => 'Русский (RU)',       'stt' => true, 'target' => true],
  'ja-JP' => ['name' => '日本語 (JP)',         'stt' => true, 'target' => true],
  'ko-KR' => ['name' => '한국어 (KR)',          'stt' => true, 'target' => true],
  'zh-CN' => ['name' => '中文（简体）',        'stt' => true, 'target' => true],
  'zh-TW' => ['name' => '中文（繁體）',        'stt' => true, 'target' => true],
];

// ––– Normalisation STT (Google v2 “latest_long/global”)
function normalize_stt_code(string $raw): string {
  $in = trim($raw);
  if ($in === '') return 'fr-FR';
  if (isset(LANG_WHITELIST[$in]) && LANG_WHITELIST[$in]['stt']) return $in;

  $map = [
    // Français
    'fr'   => 'fr-FR',
    'fr-be'=> 'fr-FR',
    'fr-ca'=> 'fr-FR',
    'fr-ch'=> 'fr-FR',
    // Anglais
    'en'   => 'en-US',
    'en-gb'=> 'en-US',
    'en-au'=> 'en-US',
    'en-ca'=> 'en-US',
    // Arabe (map variantes → ar)
    'ar-tn'=> 'ar',
    'ar-ma'=> 'ar',
    'ar-dz'=> 'ar',
    'ar-eg'=> 'ar',
  ];
  $lc = strtolower($in);
  if (isset($map[$lc])) return $map[$lc];

  $base = strtolower(explode('-', $in)[0]);
  foreach (LANG_WHITELIST as $code => $meta) {
    if ($meta['stt'] && strtolower(explode('-', $code)[0]) === $base) {
      return $code;
    }
  }
  return 'fr-FR';
}

// ––– Normalisation CIBLE (traduction + TTS)
function normalize_target_code(string $raw): string {
  $in = trim($raw);
  if ($in === '') return 'fr-FR';
  if (isset(LANG_WHITELIST[$in]) && LANG_WHITELIST[$in]['target']) return $in;

  $base = strtolower(explode('-', $in)[0]);
  foreach (LANG_WHITELIST as $code => $meta) {
    if ($meta['target'] && strtolower(explode('-', $code)[0]) === $base) {
      return $code;
    }
  }
  return 'en-US';
}

// -------------------------------
// Helper debug (facultatif)
// -------------------------------
if (!function_exists('cfg_info')) {
  function cfg_info(): array {
    $mask = function (?string $k): ?string {
      if (!$k) return null;
      $len = strlen($k);
      if ($len <= 8) return str_repeat('•', $len);
      return substr($k, 0, 4) . '…' . substr($k, -4);
    };

    return [
      'config.path' => __FILE__,
      'php.version' => PHP_VERSION,
      'php.sapi'    => PHP_SAPI,
      'env' => [
        'GOOGLE_APPLICATION_CREDENTIALS' => getenv('GOOGLE_APPLICATION_CREDENTIALS') ?: 'NOT SET',
        'GOOGLE_CLOUD_PROJECT'           => getenv('GOOGLE_CLOUD_PROJECT') ?: (getenv('GCLOUD_PROJECT') ?: 'NOT SET'),
      ],
      'dirs' => [
        'SESSIONS_DIR'   => SESSIONS_DIR,
        'TTS_PUBLIC_DIR' => TTS_PUBLIC_DIR,
        'FEED_DIR'       => FEED_DIR,
        'CACHE_DIR'      => CACHE_DIR,
      ],
      'tts' => [
        'default'          => TTS_DEFAULT_PROVIDER,
        'eleven.keySet'    => ELEVEN_API_KEY !== '',
        'eleven.keyMasked' => $mask(ELEVEN_API_KEY),
        'openai.keySet'    => OPENAI_API_KEY !== '',
        'openai.keyMasked' => $mask(OPENAI_API_KEY),
      ],
    ];
  }
}
