// public/app.js
const API_CHUNK = "../api/chunk.php";
const API_FEED  = "../api/feed.php";

const SLICE_MS  = 2500; // taille des slices en Talking verrouillé

// État
let stream;
let mediaRecorder;

// Lecture feed
let isJoined   = false;
let isListening = false;
let pollTimer  = null;
let lastSince  = 0;
const playQueue = [];
let isPlaying  = false;

// PTT
let isHolding  = false; // hold-to-talk en cours
let isLatched  = false; // talking verrouillé
let holdRec    = null;  // {mime, chunks:[], recorder}
let tapTS      = 0;     // pour détecter le double tap

// Identité/session
let sessionId;
let clientId;         // stable pour exclure sa propre TTS
let channelId;        // "12.34"
let seq = 0;
let sending = false;

// UI
const startBtn   = document.getElementById("startBtn");
const stopBtn    = document.getElementById("stopBtn");
const talkBtn    = document.getElementById("talkBtn");
const feedEl     = document.getElementById("feed");
const sourceLang = document.getElementById("sourceLang");
const targetLang = document.getElementById("targetLang");
const channelInp = document.getElementById("channel");

function log(line){ feedEl.innerHTML += line + "\n"; feedEl.scrollTop = feedEl.scrollHeight; }

function ensureClientId() {
  try {
    const k='niva.clientId';
    let id = localStorage.getItem(k);
    if (!id) { id = (crypto?.randomUUID?.() || String(Date.now())); localStorage.setItem(k, id); }
    clientId = id.replace(/[^a-z0-9-]/gi,'').toLowerCase();
  } catch {
    clientId = (crypto?.randomUUID?.() || String(Date.now())).replace(/[^a-z0-9-]/gi,'').toLowerCase();
  }
}

function pickMime() {
  const CANDIDATES = ['audio/ogg;codecs=opus','audio/webm;codecs=opus','audio/webm'];
  return CANDIDATES.find(m => window.MediaRecorder && MediaRecorder.isTypeSupported(m)) || '';
}

/* -------------------------------- JOIN / FEED -------------------------------- */

async function start() {
  const ch = parseFloat(channelInp.value);
  if (isNaN(ch) || ch < 0 || ch > 111.11) { log("ERREUR : channel invalide (0–111.11)"); return; }
  channelId = ch.toFixed(2);
  ensureClientId();

  sessionId = (crypto?.randomUUID?.() || String(Date.now()));
  seq = 0;

  // On démarre le feed à partir de "maintenant" (pas d'historique)
  isJoined = true;
  isListening = true;
  startPolling({ cursorNow:true });

  startBtn.disabled = true;
  stopBtn.disabled = false;
  talkBtn.disabled = false;
  log(`>> Rejoint le channel ${channelId} (sans historique)`);
}

function stop() {
  isJoined = false;
  isListening = false;
  if (pollTimer) { clearTimeout(pollTimer); pollTimer = null; }
  lastSince = 0;

  stopLatched(); // au cas où
  stopHold(true);

  if (stream) { try { stream.getTracks().forEach(t => t.stop()); } catch {} stream = null; }

  startBtn.disabled = false;
  stopBtn.disabled  = true;
  talkBtn.disabled  = true;
  talkBtn.classList.remove('holding','latched');
  log(">> Quitté le channel");
}

function startPolling({ cursorNow=false } = {}) {
  const poll = async () => {
    if (!isListening) return;
    try {
      let url;
      if (cursorNow) {
        // initialisation : on avance le curseur à "maintenant"
        url = `${API_FEED}?channel=${encodeURIComponent(channelId)}&cursor=now&limit=0`;
        cursorNow = false;
      } else {
        url = `${API_FEED}?channel=${encodeURIComponent(channelId)}&since=${encodeURIComponent(lastSince)}&exclude_speaker_id=${encodeURIComponent(clientId)}&exclude_session_id=${encodeURIComponent(sessionId)}&limit=50`;
      }
      const res = await fetch(url);
      const data = await res.json();
      if (data?.ok) {
        // mise à jour du curseur
        if (typeof data.next_since === 'number') lastSince = data.next_since;

        if (Array.isArray(data.items) && data.items.length) {
          for (const it of data.items) {
            // garde-fous côté client
            if (it.speaker_id && String(it.speaker_id).toLowerCase() === clientId) continue;
            if (it.session_id && String(it.session_id) === sessionId) continue;

            log(`[feed] ${it.channel} • ${it.source_lang}▶${it.target_lang}\nTRD: ${it.text}\n`);
            if (it.tts_url) playQueue.push(it.tts_url);
          }
          if (!isPlaying) playNext();
        }
      }
    } catch {}
    finally {
      pollTimer = setTimeout(poll, 900); // ~ 1s
    }
  };
  if (pollTimer) clearTimeout(pollTimer);
  pollTimer = setTimeout(poll, 50);
}

function playNext() {
  if (!isListening) return;
  if (isPlaying) return;
  const next = playQueue.shift();
  if (!next) return;
  isPlaying = true;
  const a = new Audio(next);
  a.onended = a.onerror = () => { isPlaying = false; playNext(); };
  a.play().catch(() => { isPlaying = false; playNext(); });
}

/* ------------------------------ MICRO UTILS ------------------------------ */

async function ensureMic() {
  if (stream) return stream;
  const constraints = {
    audio: {
      channelCount: 1,
      sampleRate: 48000,
      echoCancellation: true,
      noiseSuppression: false,
      autoGainControl: false
    }
  };
  stream = await navigator.mediaDevices.getUserMedia(constraints);
  return stream;
}

/* -------------------------- HOLD-TO-TALK (un seul slice) -------------------------- */

async function startHold() {
  if (!isJoined || isHolding || isLatched) return;
  await ensureMic();

  const mime = pickMime();
  holdRec = { mime, chunks: [], recorder: null };

  const rec = new MediaRecorder(stream, { mimeType: mime, audioBitsPerSecond: 128000 });
  holdRec.recorder = rec;
  isHolding = true;
  talkBtn.classList.add('holding');

  rec.ondataavailable = (e) => { if (e.data && e.data.size) holdRec.chunks.push(e.data); };
  rec.onstop = async () => {
    try {
      if (!holdRec) return;
      const blob = new Blob(holdRec.chunks, { type: holdRec.mime || 'audio/webm' });
      holdRec = null;
      await sendChunk(blob); // envoi d’un seul “slice” = tout le speech
    } finally {
      talkBtn.classList.remove('holding');
      isHolding = false;
    }
  };

  rec.start(); // pas de timeslice: un seul blob sur stop
}

function stopHold(force=false) {
  if (!holdRec || !holdRec.recorder) return;
  try {
    if (holdRec.recorder.state !== 'inactive') {
      if (!force) try { holdRec.recorder.requestData(); } catch {}
      holdRec.recorder.stop();
    }
  } catch {}
}

/* -------------------------- TALKING verrouillé (slices 4s) ------------------------- */

async function startLatched() {
  if (!isJoined || isLatched) return;
  await ensureMic();

  isLatched = true;
  talkBtn.classList.add('latched');

  const loop = () => {
    if (!isLatched) return;
    const mime = pickMime();
    const rec = new MediaRecorder(stream, { mimeType: mime, audioBitsPerSecond: 128000 });

    rec.ondataavailable = (e) => { if (e.data && e.data.size) sendChunk(e.data); };
    rec.onstop = () => { if (isLatched) setTimeout(loop, 0); };

    rec.start(); // blob autonome
    setTimeout(() => {
      try { rec.requestData(); } catch {}
      try { if (rec.state !== 'inactive') rec.stop(); } catch {}
    }, SLICE_MS);
  };
  loop();
}

function stopLatched() {
  if (!isLatched) return;
  isLatched = false;
  talkBtn.classList.remove('latched');
  // Les enregistreurs en cours s'arrêteront lors de leur timeout / stop.
}

/* --------------------------- ENVOI D’UN CHUNK --------------------------- */

async function sendChunk(blob) {
  // anti-overlap
  while (sending) { await new Promise(r => setTimeout(r, 40)); }
  sending = true;

  try {
    const fd = new FormData();
    const ext = blob.type.includes('ogg') ? 'ogg' : 'webm';
    fd.append("audio", blob, `seq-${seq}.${ext}`);
    fd.append("session_id", sessionId);
    fd.append("speaker_id", clientId);
    fd.append("channel", channelId);
    fd.append("seq", String(seq));
    fd.append("source_lang", sourceLang.value);
    fd.append("target_lang", targetLang.value);

    const res = await fetch(API_CHUNK, { method: "POST", body: fd });
    const raw = await res.text();
    let json; try { json = JSON.parse(raw); } catch { log(`ERREUR serveur (non-JSON): ${raw.slice(0,200)}...`); return; }

    const seqLabel = (json && json.seq !== null && typeof json.seq !== 'undefined') ? json.seq : '?';
    if (json.ok) {
      log(`[${seqLabel}] ${json.source_lang} ▶ ${json.target_lang}\nSTT: ${json.text}\nTRD: ${json.translated}\n`);
      // NOTE: L’orateur n’entend pas sa propre TTS (le feed l’exclut).
    } else {
      log(`ERREUR [${seqLabel}] : ${json.error || "inconnue"}`);
    }
  } catch (e) {
    log("ERREUR réseau : " + (e?.message || e));
  } finally {
    seq++; sending = false;
  }
}

/* --------------------------- GESTES SUR LE BOUTON TALK --------------------------- */

function onTalkPointerDown(ev) {
  if (!isJoined) return;
  ev.preventDefault();

  const now = Date.now();
  if (now - tapTS < 300) {
    // double tap → toggle latch
    tapTS = 0;
    if (isLatched) stopLatched(); else startLatched();
    return;
  }
  tapTS = now;

  // sinon: Hold-to-talk
  startHold();
}
function onTalkPointerUp(ev) {
  if (!isJoined) return;
  ev.preventDefault();
  // si on est en hold (non latched), on arrête
  if (!isLatched) stopHold();
}

startBtn.addEventListener("click", start);
stopBtn.addEventListener("click", stop);

talkBtn.addEventListener("pointerdown", onTalkPointerDown);
talkBtn.addEventListener("pointerup", onTalkPointerUp);
talkBtn.addEventListener("pointercancel", onTalkPointerUp);
talkBtn.addEventListener("pointerleave", onTalkPointerUp);

// Sécurité à la fermeture
window.addEventListener("beforeunload", () => { try { stop(); } catch {} });
