<?php
// api/feed.php
declare(strict_types=1);

require_once __DIR__ . '/config.php';
header('Content-Type: application/json; charset=utf-8');

function out(int $code, array $payload): never {
  http_response_code($code);
  echo json_encode($payload, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
  exit;
}
function sanitize_channel(string $raw): array {
  $raw = trim($raw);
  if ($raw === '' || !is_numeric($raw)) return [null, 'channel invalide'];
  $f = (float)$raw;
  if ($f < 0 || $f > 111.11) return [null, 'channel hors limites'];
  $norm = number_format($f, 2, '.', '');
  $file = str_replace('.', '_', $norm);
  return [$norm, $file];
}
function parse_list(string $csv): array {
  $out = [];
  foreach (explode(',', $csv) as $p) {
    $p = trim($p);
    if ($p === '') continue;
    $p = strtolower(preg_replace('/[^a-z0-9-]/i', '', $p));
    if ($p !== '') $out[$p] = true;
  }
  return $out;
}

try {
  if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

  $channel  = $_GET['channel'] ?? null;
  $cursor   = $_GET['cursor']  ?? null; // "now" pour ignorer l'historique
  $since    = isset($_GET['since']) ? (float)$_GET['since'] : 0.0;
  $excSpk   = $_GET['exclude_speaker_id'] ?? '';
  $excSess  = $_GET['exclude_session_id'] ?? '';
  $limit    = isset($_GET['limit']) ? max(0, min(100, (int)$_GET['limit'])) : 50;

  if (!$channel) out(400, ['ok'=>false,'error'=>'channel requis']);
  [$norm, $file] = sanitize_channel($channel);
  if ($norm === null) out(400, ['ok'=>false,'error'=>'channel invalide']);

  // Si on demande explicitement de se caler à "maintenant"
  if (is_string($cursor) && strtolower($cursor) === 'now') {
    out(200, ['ok'=>true, 'items'=>[], 'next_since'=>microtime(true)]);
  }

  $exSpeaker = parse_list($excSpk);
  $exSession = parse_list($excSess);

  $path = FEED_DIR . '/' . $file . '.ndjson';
  if (!is_file($path)) out(200, ['ok'=>true, 'items'=>[], 'next_since'=>$since]);

  $items = [];
  if ($limit > 0) {
    $fh = fopen($path, 'rb');
    if ($fh) {
      while (($line = fgets($fh)) !== false) {
        $line = trim($line);
        if ($line === '') continue;
        $obj = json_decode($line, true);
        if (!is_array($obj)) continue;
        if (!isset($obj['ts'])) continue;
        if ($obj['ts'] <= $since) continue;

        $sid = isset($obj['speaker_id']) ? strtolower($obj['speaker_id']) : '';
        if ($sid !== '' && isset($exSpeaker[$sid])) continue;

        $sess = isset($obj['session_id']) ? strtolower($obj['session_id']) : '';
        if ($sess !== '' && isset($exSession[$sess])) continue;

        $items[] = $obj;
        if (count($items) >= $limit) break;
      }
      fclose($fh);
    }
  }

  $next = $since;
  foreach ($items as $it) if ($it['ts'] > $next) $next = $it['ts'];

  out(200, ['ok'=>true, 'items'=>$items, 'next_since'=>$next]);

} catch (Throwable $e) {
  out(500, ['ok'=>false, 'error'=>'Exception: '.$e->getMessage()]);
}
