<?php
// api/channel_settings.php — gestion serveur du "host" + réglages de channel
declare(strict_types=1);

ini_set('display_errors', '1');
error_reporting(E_ALL);
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Headers: Content-Type');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');

require_once __DIR__ . '/config.php';

function out(int $code, array $payload): never {
  http_response_code($code);
  echo json_encode($payload, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
  exit;
}
function sanitize_channel(string $raw): array {
  $raw = trim($raw);
  if ($raw === '' || !is_numeric($raw)) return [null, null, 'channel invalide'];
  $f = (float)$raw;
  if ($f < 0 || $f > 111.11) return [null, null, 'channel hors limites'];
  $norm = number_format($f, 2, '.', '');
  $file = str_replace('.', '_', $norm);
  return [$norm, $file, null];
}
function feed_base(): string {
  if (!defined('FEED_DIR')) define('FEED_DIR', __DIR__ . '/feeds');
  @mkdir(FEED_DIR, 0775, true);
  return FEED_DIR;
}
function settings_path(string $channelFile): string {
  return feed_base() . '/' . $channelFile . '.settings.json';
}
function read_settings(string $path): array {
  if (!is_file($path)) {
    return [
      'allow_listener_talk' => 1,
      'host_session_id'     => null,
      'host_speaker_id'     => null,
      'host_last_seen'      => 0,
    ];
  }
  $j = json_decode((string)@file_get_contents($path), true);
  if (!is_array($j)) $j = [];
  $j += [
    'allow_listener_talk' => 1,
    'host_session_id'     => null,
    'host_speaker_id'     => null,
    'host_last_seen'      => 0,
  ];
  return $j;
}
function write_settings(string $path, array $data): bool {
  $tmp = $path . '.tmp';
  $ok = @file_put_contents($tmp, json_encode($data, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES|JSON_PRETTY_PRINT));
  if ($ok === false) return false;
  return @rename($tmp, $path);
}
function is_current_host(array $s, ?string $sessId, ?string $spkId): bool {
  if (!$sessId || !$spkId) return false;
  return (strtolower((string)$s['host_session_id']) === strtolower($sessId))
      && (strtolower((string)$s['host_speaker_id']) === strtolower($spkId));
}

/* ---------------------------- ROUTAGE ---------------------------- */

$method = $_SERVER['REQUEST_METHOD'] ?? 'GET';
if ($method === 'OPTIONS') { http_response_code(204); exit; }

$channelRaw = $_GET['channel'] ?? $_POST['channel'] ?? null;
if (!$channelRaw) out(400, ['ok'=>false,'error'=>'channel requis']);
[$channelNorm, $channelFile, $chanErr] = sanitize_channel($channelRaw);
if ($chanErr !== null) out(400, ['ok'=>false,'error'=>'channel invalide (0..111.11)']);

$path = settings_path($channelFile);

if ($method === 'GET') {
  $s = read_settings($path);
  // ping host_last_seen si le client GET fournit son identité
  $sid = $_GET['session_id'] ?? null;
  $spk = $_GET['speaker_id'] ?? null;
  if ($sid && $spk && is_current_host($s, $sid, $spk)) {
    $s['host_last_seen'] = microtime(true);
    @write_settings($path, $s);
  }
  out(200, ['ok'=>true, 'channel'=>$channelNorm, 'settings'=>$s, 'ts'=>microtime(true)]);
}

if ($method === 'POST') {
  $action   = strtolower($_POST['action'] ?? '');
  $sid      = $_POST['session_id'] ?? null;
  $spk      = $_POST['speaker_id'] ?? null;
  $s        = read_settings($path);

  if ($action === 'claim_host') {
    // 1) Si pas d’hôte → on attribue
    if (!$s['host_session_id'] || !$s['host_speaker_id']) {
      $s['host_session_id'] = $sid;
      $s['host_speaker_id'] = $spk;
      $s['host_last_seen']  = microtime(true);
      if (!write_settings($path, $s)) out(500, ['ok'=>false,'error'=>'écriture settings échouée']);
      out(200, ['ok'=>true,'channel'=>$channelNorm,'settings'=>$s,'claimed'=>true]);
    }
    // 2) Si déjà hôte → on dit qui est hôte
    $isMine = is_current_host($s, $sid, $spk);
    out(200, ['ok'=>true,'channel'=>$channelNorm,'settings'=>$s,'claimed'=>$isMine]);
  }

  if ($action === 'release_host') {
    if (is_current_host($s, $sid, $spk)) {
      $s['host_session_id'] = null; $s['host_speaker_id'] = null; $s['host_last_seen'] = 0;
      if (!write_settings($path, $s)) out(500, ['ok'=>false,'error'=>'écriture settings échouée']);
      out(200, ['ok'=>true,'channel'=>$channelNorm,'settings'=>$s,'released'=>true]);
    }
    out(403, ['ok'=>false,'error'=>"vous n'êtes pas l'hôte"]);
  }

  if ($action === 'set') {
    // Seul l’hôte reconnu peut modifier
    if (!is_current_host($s, $sid, $spk)) out(403, ['ok'=>false,'error'=>"vous n'êtes pas l'hôte"]);

    if (isset($_POST['allow_listener_talk'])) {
      $s['allow_listener_talk'] = ((int)$_POST['allow_listener_talk']) ? 1 : 0;
    }
    $s['host_last_seen'] = microtime(true);

    if (!write_settings($path, $s)) out(500, ['ok'=>false,'error'=>'écriture settings échouée']);
    out(200, ['ok'=>true,'channel'=>$channelNorm,'settings'=>$s]);
  }

  // compat (anciens appels) : POST sans action = set allow_listener_talk si hôte
  if (isset($_POST['allow_listener_talk'])) {
    if (!is_current_host($s, $sid, $spk)) out(403, ['ok'=>false,'error'=>"vous n'êtes pas l'hôte"]);
    $s['allow_listener_talk'] = ((int)$_POST['allow_listener_talk']) ? 1 : 0;
    $s['host_last_seen'] = microtime(true);
    if (!write_settings($path, $s)) out(500, ['ok'=>false,'error'=>'écriture settings échouée']);
    out(200, ['ok'=>true,'channel'=>$channelNorm,'settings'=>$s]);
  }

  out(400, ['ok'=>false,'error'=>'action manquante ou invalide']);
}

out(405, ['ok'=>false,'error'=>'Méthode non supportée']);
