<?php
// api/channel_mute.php — mute/unmute d’un auditeur (par speaker_id ou session_id)
declare(strict_types=1);

ini_set('display_errors', '1');
error_reporting(E_ALL);
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Headers: Content-Type');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');

require_once __DIR__ . '/config.php';

function out(int $code, array $payload): never {
  http_response_code($code);
  echo json_encode($payload, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
  exit;
}
function sanitize_channel(string $raw): array {
  $raw = trim($raw);
  if ($raw === '' || !is_numeric($raw)) return [null, null, 'channel invalide'];
  $f = (float)$raw;
  if ($f < 0 || $f > 111.11) return [null, null, 'channel hors limites'];
  $norm = number_format($f, 2, '.', '');
  $file = str_replace('.', '_', $norm);
  return [$norm, $file, null];
}
function feed_base(): string {
  if (!defined('FEED_DIR')) define('FEED_DIR', __DIR__ . '/feeds');
  @mkdir(FEED_DIR, 0775, true);
  return FEED_DIR;
}
function mute_path(string $channelFile): string {
  return feed_base() . '/' . $channelFile . '.mutes.json';
}
function read_mutes(string $path): array {
  if (!is_file($path)) return ['speakers'=>[], 'sessions'=>[]];
  $j = json_decode((string)@file_get_contents($path), true);
  if (!is_array($j)) return ['speakers'=>[], 'sessions'=>[]];
  $j += ['speakers'=>[], 'sessions'=>[]];
  return $j;
}
function write_mutes(string $path, array $data): bool {
  $tmp = $path . '.tmp';
  $ok = @file_put_contents($tmp, json_encode($data, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES|JSON_PRETTY_PRINT));
  if ($ok === false) return false;
  return @rename($tmp, $path);
}

$method = $_SERVER['REQUEST_METHOD'] ?? 'GET';
if ($method === 'OPTIONS') { http_response_code(204); exit; }

$channelRaw = $_GET['channel'] ?? $_POST['channel'] ?? null;
if (!$channelRaw) out(400, ['ok'=>false,'error'=>'channel requis']);
[$channelNorm, $channelFile, $chanErr] = sanitize_channel($channelRaw);
if ($chanErr !== null) out(400, ['ok'=>false,'error'=>'channel invalide (0..111.11)']);

$path = mute_path($channelFile);

if ($method === 'GET') {
  $m = read_mutes($path);
  out(200, ['ok'=>true, 'channel'=>$channelNorm, 'mutes'=>$m, 'ts'=>microtime(true)]);
}

if ($method === 'POST') {
  $action = strtolower($_POST['action'] ?? '');
  $type   = strtolower($_POST['type'] ?? 'speaker'); // speaker|session
  $id     = strtolower(preg_replace('/[^a-z0-9-]/i','', $_POST['id'] ?? ''));

  if (!in_array($action, ['add','remove','toggle'], true)) out(400, ['ok'=>false,'error'=>'action invalide']);
  if (!in_array($type, ['speaker','session'], true)) out(400, ['ok'=>false,'error'=>'type invalide']);
  if ($id === '') out(400, ['ok'=>false,'error'=>'id vide']);

  $m = read_mutes($path);
  $key = $type === 'speaker' ? 'speakers' : 'sessions';
  $arr = $m[$key];

  $has = in_array($id, $arr, true);

  if ($action === 'add' || ($action === 'toggle' && !$has)) {
    $arr[] = $id;
  } elseif ($action === 'remove' || ($action === 'toggle' && $has)) {
    $arr = array_values(array_filter($arr, fn($x) => $x !== $id));
  }

  $m[$key] = array_values(array_unique($arr));
  if (!write_mutes($path, $m)) out(500, ['ok'=>false,'error'=>'écriture mutes échouée']);

  out(200, ['ok'=>true, 'channel'=>$channelNorm, 'mutes'=>$m, 'ts'=>microtime(true)]);
}

out(405, ['ok'=>false,'error'=>'Méthode non supportée']);
