// public/app.js
const API_CHUNK = "../api/chunk.php";
const API_FEED  = "../api/feed.php";
const SLICE_MS  = 5000; // durée d'un chunk indépendant

let stream;
let mediaRecorder;
let isCapturing = false;
let isListening = false;

let sessionId;
let clientId;       // persistant pour exclure sa propre voix
let channelId;      // "12.34" string normalisée
let seq = 0;
let sending = false;

let pollTimer = null;
let lastSince = 0;
const playQueue = [];
let isPlaying = false;

// UI
const startBtn   = document.getElementById("startBtn");
const stopBtn    = document.getElementById("stopBtn");
const feedEl     = document.getElementById("feed");
const sourceLang = document.getElementById("sourceLang");
const targetLang = document.getElementById("targetLang");
const roleSel    = document.getElementById("role");
const channelInp = document.getElementById("channel");

function log(line) {
  feedEl.innerHTML += line + "\n";
  feedEl.scrollTop = feedEl.scrollHeight;
}

function pickMime() {
  const CANDIDATES = ['audio/ogg;codecs=opus', 'audio/webm;codecs=opus', 'audio/webm'];
  return CANDIDATES.find(m => window.MediaRecorder && MediaRecorder.isTypeSupported(m)) || '';
}

function ensureClientId() {
  try {
    const k = 'niva.clientId';
    let id = localStorage.getItem(k);
    if (!id) { id = (crypto && crypto.randomUUID) ? crypto.randomUUID() : String(Date.now()); localStorage.setItem(k, id); }
    clientId = id.replace(/[^a-z0-9-]/gi, '').toLowerCase();
  } catch {
    clientId = (crypto && crypto.randomUUID) ? crypto.randomUUID() : String(Date.now());
  }
}

async function start() {
  // Valide le channel
  const ch = parseFloat(channelInp.value);
  if (isNaN(ch) || ch < 0 || ch > 111.11) { log("ERREUR : channel invalide (0–111.11)"); return; }
  channelId = ch.toFixed(2); // "12.34"

  ensureClientId();
  sessionId = (crypto && crypto.randomUUID) ? crypto.randomUUID() : String(Date.now());
  seq = 0;
  lastSince = 0;

  const role = roleSel.value;

  try {
    if (role === 'speaker') {
      // Ouvre le micro
      stream = await navigator.mediaDevices.getUserMedia({
        audio: {
          channelCount: 1,
          sampleRate: 48000,
          echoCancellation: true,
          noiseSuppression: false,  // moins agres.
          autoGainControl: false
        }
      });
      isCapturing = true;
      startOneSlice();
    }

    // Toujours écouter le channel (même orateur, mais on exclut self)
    isListening = true;
    startPolling();

    startBtn.disabled = true;
    stopBtn.disabled  = false;
    log(`>> Rejoint le channel ${channelId} en mode ${role === 'speaker' ? 'Orateur' : 'Auditeur'}`);
  } catch (err) {
    log("ERREUR initialisation : " + (err?.message || err));
  }
}

function stop() {
  isCapturing = false;
  isListening = false;

  try { if (pollTimer) { clearTimeout(pollTimer); pollTimer = null; } } catch {}
  lastSince = 0;

  try { if (mediaRecorder && mediaRecorder.state !== "inactive") mediaRecorder.stop(); } catch {}
  try { if (stream) stream.getTracks().forEach(t => t.stop()); } catch {}
  stream = null;

  startBtn.disabled = false;
  stopBtn.disabled  = true;
  log(">> Quitté le channel");
}

// Démarre un enregistreur, l'arrête après SLICE_MS, et enchaîne
function startOneSlice() {
  if (!isCapturing) return;

  const mime = pickMime();
  mediaRecorder = new MediaRecorder(stream, { mimeType: mime, audioBitsPerSecond: 128000 });
  console.log('[MediaRecorder] mime =', mediaRecorder.mimeType);

  mediaRecorder.ondataavailable = (e) => {
    if (e.data && e.data.size > 0) sendChunk(e.data);
  };

  mediaRecorder.onstop = () => {
    if (isCapturing) setTimeout(startOneSlice, 0);
  };

  mediaRecorder.start(); // blob autonome à chaque stop

  setTimeout(() => {
    if (!mediaRecorder) return;
    try { mediaRecorder.requestData(); } catch {}
    try { if (mediaRecorder.state !== "inactive") mediaRecorder.stop(); } catch {}
  }, SLICE_MS);
}

async function sendChunk(blob) {
  while (sending) { await new Promise(r => setTimeout(r, 50)); }
  sending = true;

  try {
    const fd = new FormData();
    const ext = blob.type.includes('ogg') ? 'ogg' : 'webm';
    fd.append("audio", blob, `seq-${seq}.${ext}`);
    fd.append("session_id", sessionId);
    fd.append("speaker_id", clientId); // pour exclure self
    fd.append("channel", channelId);
    fd.append("seq", String(seq));
    fd.append("source_lang", sourceLang.value);
    fd.append("target_lang", targetLang.value);

    const res = await fetch(API_CHUNK, { method: "POST", body: fd });
    const raw = await res.text();
    let json;
    try { json = JSON.parse(raw); }
    catch { log(`ERREUR serveur (non-JSON): ${raw.slice(0, 200)}...`); return; }

    const seqLabel = (json && typeof json.seq !== 'undefined' && json.seq !== null) ? json.seq : '?';

    if (json.ok) {
      // L'orateur NE joue PAS sa propre TTS ici.
      // Tout le monde (y compris l'orateur) entend via le feed.
      const line = `[${seqLabel}] ${json.source_lang} ▶ ${json.target_lang}\n` +
                   `STT: ${json.text}\n` +
                   `TRD: ${json.translated}\n`;
      log(line);
    } else {
      log(`ERREUR [${seqLabel}] : ${json.error || "inconnue"}`);
    }
  } catch (e) {
    log("ERREUR réseau : " + (e?.message || e));
  } finally {
    seq++;
    sending = false;
  }
}

function startPolling() {
  const poll = async () => {
    if (!isListening) return;
    try {
      const url = `${API_FEED}?channel=${encodeURIComponent(channelId)}&since=${encodeURIComponent(lastSince)}&exclude_speaker_id=${encodeURIComponent(clientId)}&limit=25`;
      const res = await fetch(url);
      const data = await res.json();
      if (data?.ok && Array.isArray(data.items)) {
        for (const it of data.items) {
          // Affiche le texte + programme la lecture
          const line = `[feed] ${it.channel} • ${it.source_lang}▶${it.target_lang}\nTRD: ${it.text}\n`;
          log(line);
          if (it.tts_url) playQueue.push(it.tts_url);
        }
        if (typeof data.next_since === 'number') lastSince = data.next_since;
        if (!isPlaying) playNext();
      }
    } catch (e) {
      // silencieux
    } finally {
      pollTimer = setTimeout(poll, 1200); // ~1.2s
    }
  };
  if (pollTimer) clearTimeout(pollTimer);
  pollTimer = setTimeout(poll, 100);
}

function playNext() {
  if (!isListening) return;
  if (isPlaying) return;
  const next = playQueue.shift();
  if (!next) return;
  isPlaying = true;
  const audio = new Audio(next);
  audio.onended = () => { isPlaying = false; playNext(); };
  audio.onerror = () => { isPlaying = false; playNext(); };
  audio.play().catch(() => { isPlaying = false; playNext(); });
}

startBtn.addEventListener("click", start);
stopBtn.addEventListener("click", stop);

window.addEventListener("beforeunload", () => {
  try { if (mediaRecorder && mediaRecorder.state === "recording") mediaRecorder.stop(); } catch {}
  try { if (stream) stream.getTracks().forEach(t => t.stop()); } catch {}
  try { if (pollTimer) clearTimeout(pollTimer); } catch {}
});
