// public/app.js
const API_URL = "../api/chunk.php";
const SLICE_MS = 5000; // durée d'un chunk

let stream;
let mediaRecorder;
let isCapturing = false;
let sessionId;
let seq = 0;
let sending = false;

const startBtn   = document.getElementById("startBtn");
const stopBtn    = document.getElementById("stopBtn");
const feed       = document.getElementById("feed");
const sourceLang = document.getElementById("sourceLang");
const targetLang = document.getElementById("targetLang");

// Meilleur MIME disponible (on préfère OGG/Opus, puis WebM/Opus)
function pickMime() {
  const CANDIDATES = ['audio/ogg;codecs=opus', 'audio/webm;codecs=opus', 'audio/webm'];
  return CANDIDATES.find(m => window.MediaRecorder && MediaRecorder.isTypeSupported(m)) || '';
}

function log(line) {
  feed.innerHTML += line + "\n";
  feed.scrollTop = feed.scrollHeight;
}

async function start() {
  try {
    if (!navigator.mediaDevices?.getUserMedia) {
      log("ERREUR : getUserMedia non supporté.");
      return;
    }
    if (typeof MediaRecorder === "undefined") {
      log("ERREUR : MediaRecorder non supporté.");
      return;
    }

    // 1) Prépare la session
    sessionId = (crypto && crypto.randomUUID) ? crypto.randomUUID() : String(Date.now());
    seq = 0;

    // 2) Ouvre le micro avec contraintes propres (mono, 48 kHz)
    stream = await navigator.mediaDevices.getUserMedia({
      audio: {
        channelCount: 1,
        sampleRate: 48000,
        echoCancellation: true,
        noiseSuppression: true,
        autoGainControl: true
      }
    });

    isCapturing = true;
    startBtn.disabled = true;
    stopBtn.disabled = false;

    log(">> Capture démarrée");
    // 3) Lance la première tranche
    startOneSlice();
  } catch (err) {
    log("ERREUR permission micro : " + (err?.message || err));
  }
}

// Démarre un enregistreur, l'arrête après SLICE_MS, et recommence
function startOneSlice() {
  if (!isCapturing) return;

  const mime = pickMime();
  mediaRecorder = new MediaRecorder(stream, { mimeType: mime, audioBitsPerSecond: 128000 });
  console.log('[MediaRecorder] mime =', mediaRecorder.mimeType);

  mediaRecorder.ondataavailable = (e) => {
    if (e.data && e.data.size > 0) {
      sendChunk(e.data);
    }
  };

  mediaRecorder.onstop = () => {
    if (isCapturing) {
      // Enchaîne immédiatement la tranche suivante
      setTimeout(startOneSlice, 0);
    }
  };

  mediaRecorder.start(); // pas de timeslice: on force un blob autonome par stop()
  // Arrêt programmé : on force un flush propre même si peu de données
  setTimeout(() => {
    if (!mediaRecorder) return;
    try { mediaRecorder.requestData(); } catch {}
    try { if (mediaRecorder.state !== "inactive") mediaRecorder.stop(); } catch {}
  }, SLICE_MS);
}

function stop() {
  isCapturing = false;
  try {
    if (mediaRecorder && mediaRecorder.state !== "inactive") {
      mediaRecorder.stop();
    }
  } catch {}
  try {
    if (stream) {
      stream.getTracks().forEach(t => t.stop());
    }
  } catch {}

  startBtn.disabled = false;
  stopBtn.disabled  = true;
  log(">> Capture arrêtée");
}

async function sendChunk(blob) {
  // Évite les overlaps d'envois
  while (sending) { await new Promise(r => setTimeout(r, 50)); }
  sending = true;

  try {
    const fd = new FormData();
    const ext = blob.type.includes('ogg') ? 'ogg' : 'webm';
    fd.append("audio", blob, `seq-${seq}.${ext}`);
    fd.append("session_id", sessionId);
    fd.append("seq", String(seq));
    fd.append("source_lang", sourceLang.value);
    fd.append("target_lang", targetLang.value);

    const res = await fetch(API_URL, { method: "POST", body: fd });
    const raw = await res.text();

    let json;
    try { json = JSON.parse(raw); }
    catch {
      log(`ERREUR serveur (non-JSON): ${raw.slice(0, 200)}...`);
      return;
    }

    const seqLabel = (json && typeof json.seq !== 'undefined' && json.seq !== null) ? json.seq : '?';

    if (json.ok) {
      const line = `[${seqLabel}] ${json.source_lang} ▶ ${json.target_lang}\n` +
                   `STT: ${json.text}\n` +
                   `TRD: ${json.translated}\n`;
      log(line);
      if (json.tts_url) { try { new Audio(json.tts_url).play(); } catch {} }
    } else {
      log(`ERREUR [${seqLabel}] : ${json.error || "inconnue"}`);
    }
  } catch (e) {
    log("ERREUR réseau : " + (e?.message || e));
  } finally {
    seq++;
    sending = false;
  }
}

startBtn.addEventListener("click", start);
stopBtn.addEventListener("click", stop);

// Optionnel: sécurité à la fermeture
window.addEventListener("beforeunload", () => {
  try { if (mediaRecorder && mediaRecorder.state === "recording") mediaRecorder.stop(); } catch {}
  try { if (stream) stream.getTracks().forEach(t => t.stop()); } catch {}
});
